<?php
class RequestModel extends CI_Model {
	
	public $limit_entries = 20;
	
	function __construct() {
		parent::__construct();
		$this->load->database();
	}

	public function create_request($user_id, $type, $amount, $address, $proof = '', $note = '', $time = '') {
		$user_id = $this->db->escape_str($user_id);
		$type = $this->db->escape_str($type);
		$amount = $this->db->escape_str($amount);
		$address = $this->db->escape_str($address);
		$proof = $this->db->escape_str($proof);
		$status = '';
		$BNB_price =  (float)round($this->Statistic->get_statistic_by_date('BNB_price') / 100, 2);
		$USD_value = $BNB_price * $amount / 100000000;
		$date = date('Y-m-d');
		if($time == ''){
			$time = time();
		}
		if($type == 'Deposit') {
			$status = 'Completed';
			if (strlen($proof) < 30) {
				$status = 'Processing';
			}
			$data = array(
				'user_id' 		=> $user_id,
				'type' 			=> $type,
				'amount'		=> $amount,
				'USD_value'		=> $USD_value,
				'address'		=> $address,
				'proof'			=> $proof,
				'created_time'	=> $time,
				'completion_time'	=> $time,
				'date'			=> $date,
				'status'		=> $status,
			);
		} else if($type == 'Withdrawal') {
			$status = 'Unconfirmed';
			$data = array(
				'user_id' 		=> $user_id,
				'type' 			=> $type,
				'amount'		=> $amount,
				'USD_value'		=> $USD_value,
				'address'		=> $address,
				'created_time'	=> $time,
				'date'			=> $date,
				'status'		=> $status,
				'note'			=> $note,
			);
		}
		
		$this->db->insert('ez_requests', $data);
		$insert_id = $this->db->insert_id();
		
		//sync_data
		$this->auto_sync_request_data($insert_id);
		//sync_data
		
		return  $insert_id;
	}
	
	public function get_request_info_by_id($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$request_id = intval($request_id);
		$sql = "SELECT * FROM `ez_requests` WHERE `request_id` = ? LIMIT 1";
		$query = $this->db->query($sql, array($request_id));
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_request_info_by_id_for_update($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$request_id = intval($request_id);

		$sql = "SELECT * FROM `ez_requests` WHERE `request_id` = ? LIMIT 1 FOR UPDATE";
		$query = $this->db->query($sql, array($request_id));

		if ($query->num_rows() > 0) {
			return $query->result()[0];
		}

		return null;
	}
	
	public function get_all_requests_by_txid($txid) {
		$txid = $this->db->escape_str($txid);
		$sql = "SELECT * FROM `ez_requests` WHERE `proof` = ? AND `note` != 'f'";
		$query = $this->db->query($sql, array($txid));
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}
	
	public function complete_deposit_request($request_id, $USD_value = 0) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Completed');
		$this->db->set('completion_time', time(), FALSE);
		$this->db->set('confirmations', 'confirmations+1', FALSE);
		$this->db->set('note', $USD_value, FALSE);
		$this->db->where('request_id', $request_id);
		$this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
	}

	public function confirm_request($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('confirmations', 'confirmations+1', FALSE);
		$this->db->where('request_id', $request_id);
		$this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
	}
	
	//process withdrawal request
	public function process_withdrawal_request($request_id, $txid, $USD_value = 0) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Processing');
		$this->db->set('completion_time', time(), FALSE);
		$this->db->set('proof', $txid);
		$this->db->set('confirmations', 0);
		$this->db->set('note', $USD_value, TRUE);
		$this->db->where('request_id', $request_id);
		$res = $this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
		return $res;
	}
	
	public function reverse_withdrawal_request($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Pending');
		$this->db->set('completion_time', 0, FALSE);
		$this->db->where('request_id', $request_id);
		$res =  $this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
		return $res;
		
	}
	
	public function complete_withdrawal_request($request_id, $txid = '', $USD_value = 0) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Completed');
		$this->db->set('completion_time', time(), FALSE);
		if($txid != '') {
			$this->db->set('proof', $txid);
		}
		if($USD_value != 0) {
			$this->db->set('note', $USD_value, TRUE);
		}
		$this->db->set('confirmations', 1);
		$this->db->where('request_id', $request_id);
		$res =  $this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
		return $res;
		
	}
	
	public function get_oldest_processing_request() {
		$sql = "SELECT * FROM `ez_requests` WHERE `status` = 'Processing' AND `completion_time` < UNIX_TIMESTAMP() - 300 ORDER BY `request_id` ASC LIMIT 1";
		$query = $this->db->query($sql);
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_random_processing_request() {
		$sql = "SELECT * FROM `ez_requests` WHERE `type` = 'Withdrawal' AND `status` = 'Processing' AND `completion_time` < UNIX_TIMESTAMP() - 60 ORDER BY RAND() LIMIT 1";
		$query = $this->db->query($sql);
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	//process withdrawal request

	
	public function cancel_withdrawal_request($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Canceled');
		$this->db->where('request_id', $request_id);
		$res =  $this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
		return $res;
		
	}
	
	public function confirm_withdrawal_request($request_id) {
		$request_id = $this->db->escape_str($request_id);
		$this->db->set('status', 'Pending');
		$this->db->where('request_id', $request_id);
		$res =  $this->db->update('ez_requests');
		
		//sync_data
		$this->auto_sync_request_data($request_id);
		//sync_data
		
		return $res;
		
	}
	
	public function check_multiple_withdrawal_requests($user_id) {
		$user_id = $this->db->escape_str($user_id);
		$sql = "SELECT `request_id` FROM `ez_requests` WHERE `user_id` = ? AND `type` = 'Withdrawal' AND `created_time` > '".(time() - 5)."'";
		$query = $this->db->query($sql, array($user_id));
		if($query->num_rows() > 0) {
			return 0;
		}
		return 1;
	}
	
	/*public function load_user_transactions($user_id, $page_id = 1, $type = 'All', $status = 'All', $hidden_internal_tx = 0) {
		$user_id = $this->db->escape_str($user_id);
		if($hidden_internal_tx == 0) {
			if($user_id != 0) {
				if($type == 'Deposit') {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND `type` = 'Deposit' AND `amount` > 10 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					$query = $this->db->query($sql, array($user_id));
				} else if($type == 'Withdrawal') {
					if($status == 'All') {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND `type` = 'Withdrawal' AND (`status` != 'Canceled' || `created_time` > UNIX_TIMESTAMP() - 86400 * 7) ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
						$query = $this->db->query($sql, array($user_id));
					} else {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND `type` = 'Withdrawal' AND `status` = ? ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
						$query = $this->db->query($sql, array($user_id, $status));
					}
				} else {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					$query = $this->db->query($sql, array($user_id));
				}
			} else {
				if($type == 'Deposit') {
					if($status != 'All') {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `type` = 'Deposit' AND `status` = '".$status."' ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					} else {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `type` = 'Deposit' ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					}
				
				} else if($type == 'Withdrawal') {
					if($status != 'All') {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`disable_payout`='0' AND `type` = 'Withdrawal' AND `status` = '".$status."'  AND `note` != 'f' ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					} else {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`disable_payout`='0' AND `type` = 'Withdrawal' AND `status` != 'Unconfirmed'  AND `note` != 'f' AND (`status` != 'Canceled' || `created_time` > UNIX_TIMESTAMP() - 86400 * 7) ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					}
					
				} else {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				}
				$query = $this->db->query($sql);
			}
		} else {
			if($user_id != 0) {
				if($type == 'Deposit') {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND `type` = 'Deposit' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				} else if($type == 'Withdrawal') {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND `type` = 'Withdrawal' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				} else {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_requests`.`user_id` = ? AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				}
				$query = $this->db->query($sql, array($user_id));
			} else {
				if($type == 'Deposit') {
					if($status != 'All') {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `type` = 'Deposit' AND `status` = '".$status."' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					} else {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `type` = 'Deposit' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					}
				
				} else if($type == 'Withdrawal') {
					if($status != 'All') {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`disable_payout`='0' AND `type` = 'Withdrawal' AND `status` = '".$status."'  AND `note` != 'f' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					} else {
						$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `total_referrals`, `total_deposits`, `total_withdrawals`, `note`, `country`, `auto_payout` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`disable_payout`='0' AND `type` = 'Withdrawal' AND `status` != 'Unconfirmed'  AND `note` != 'f' AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 AND (`status` != 'Canceled' || `created_time` > UNIX_TIMESTAMP() - 86400 * 7) ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
					}
					
				} else {
					$sql = "SELECT `request_id` AS `id`, `ez_requests`.`user_id` AS `uid`, `type`, `amount`, `USD_value`, `address`, `proof` AS `txid`, `created_time` AS `time`, `status`, `user_name`, `note`, `country` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND CHAR_LENGTH(`proof`) = 66 AND `amount` > 10000 ORDER BY `request_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				}
				$query = $this->db->query($sql);
			}
		}
		
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}*/
	
	public function load_user_transactions($user_id, $page_id = 1, $type = 'All', $status = 'All', $hidden_internal_tx = 0) {
		$user_id = $this->db->escape_str($user_id);
		$limit = $this->limit_entries;
		$offset = $limit * ($page_id - 1);

		// Base fields to select
		$common_fields = [
			"`request_id` AS `id`", "`ez_requests`.`user_id` AS `uid`", "`type`", "`amount`", "`USD_value`",
			"`address`", "`proof` AS `txid`", "`created_time` AS `time`", "`status`", "`user_name`",
			"`note`", "`ip_address`", "`confirmations`"
		];

		// Additional fields for Withdrawal type
		$withdraw_fields = [
			"`total_referrals`", "`total_deposits`", "`total_withdrawals`", "`auto_payout`"
		];

		// If type is Withdrawal, include extra columns
		if ($type === 'Withdrawal') {
			$common_fields = array_merge($common_fields, $withdraw_fields);
		}

		// Start building SQL
		$base_sql = "SELECT " . implode(", ", $common_fields) . " FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id`";

		// Dynamic WHERE conditions
		$where = [];

		// Filter by user ID if provided
		if ($user_id != 0) {
			$where[] = "`ez_requests`.`user_id` = " . $this->db->escape($user_id);
		}

		// Filter by type (Deposit/Withdrawal)
		if (in_array($type, ['Deposit', 'Withdrawal'])) {
			$where[] = "`type` = " . $this->db->escape($type);
		}

		// Only apply special filters for public Withdrawal listing
		if ($type === 'Withdrawal' && $user_id == 0) {
			$where[] = "`ez_users`.`banned` = '0'";
			$where[] = "`ez_users`.`disable_payout` = '0'";
			$where[] = "`note` != 'f'";
		}
		
		if ($status !== 'All') {
			if ($status === 'Unconfirmed' && $type == 'Deposit') {
				$where[] = "`status` = 'Processing'";
				$where[] = "`confirmations` = 0";
			} else if ($status === 'Unconfirmed' && $type == 'Withdrawal') {
				$where[] = "`status` = 'Unconfirmed'";
			} else {
				$where[] = "`status` = " . $this->db->escape($status);
			}
		} else {
			// Show recent withdrawals even if status is All
			if($type === 'Withdrawal') {
				$where[] = "(`status` != 'Canceled' OR `created_time` > UNIX_TIMESTAMP() - 86400 * 7)";
				if($user_id == 0) {
					$where[] = "`status` != 'Unconfirmed'";
				}
			}
		}

		// Internal transaction filtering (for admin/mod view)
		if ($hidden_internal_tx) {
			$where[] = "CHAR_LENGTH(`proof`) > 30";
			$where[] = "`amount` > 10000";
		} else if ($type === 'Deposit' && $user_id != 0) {
			$where[] = "`amount` > 10";
		}

		// Finalize SQL with WHERE and pagination
		$sql = $base_sql;
		if (!empty($where)) {
			$sql .= " AND " . implode(" AND ", $where);
		}

		$sql .= " ORDER BY `request_id` DESC LIMIT $limit OFFSET $offset";

		// Execute and return results
		$query = $this->db->query($sql);
		return $query->num_rows() > 0 ? $query->result() : null;
	}
	
	public function load_payouts($page_id) {
		$sql = "SELECT user_name, amount, request_id, address, created_time, completion_time, proof, status FROM ez_requests JOIN ez_users ON ez_requests.user_id = ez_users.user_id WHERE ez_requests.type = 'Withdrawal' AND `status` = 'Completed' ORDER BY `ez_requests`.`completion_time` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
		$query = $this->db->query($sql);
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}
	
	public function get_number_transactions($user_id, $type = 'All') {
		$user_id = $this->db->escape_str($user_id);
		if($user_id != 0) {
			if($type == 'Deposit') {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests` WHERE `user_id` = ? AND `type` = 'Deposit'";
			} else if($type == 'Withdrawal') {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests` WHERE `user_id` = ? AND `type` = 'Withdrawal'";
			} else {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests` WHERE `user_id` = ?";
			}
			$query = $this->db->query($sql, array($user_id));
		} else {
			if($type == 'Deposit') {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests` WHERE `type` = 'Deposit'";
			} else if($type == 'Withdrawal') {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests` WHERE `type` = 'Withdrawal'";
			} else {
				$sql = "SELECT COUNT(`request_id`) AS `num` FROM `ez_requests`";
			}
			$query = $this->db->query($sql);
		}
		
		if($query->num_rows() > 0) {
			return $query->result()[0]->num;
		}
		return 0;
	}

	public function get_request_statistics($user_id, $type = 'All', $status = 'All') {
		// Sanitize inputs
		$user_id = $this->db->escape_str($user_id);
		$type = $this->db->escape_str($type);
		$status = $this->db->escape_str($status);

		// Build cache key from parameters
		$cache_key = 'request_stats_' . md5("uid:$user_id|type:$type|status:$status");

		// Try to get from cache
		$cached = $this->cache->get($cache_key);
		if ($cached !== false) {
			return $cached;
		}

		// Build WHERE clause
		$where = [];
		$params = [];

		if ($user_id != 0) {
			$where[] = '`user_id` = ?';
			$params[] = $user_id;
		}

		if ($type != 'All') {
			$where[] = '`type` = ?';
			$params[] = $type;
		}

		if ($status != 'All') {
			$where[] = '`status` = ?';
			$params[] = $status;
		}

		// SQL query
		$sql = "SELECT COUNT(*) AS `total_requests`, SUM(`amount`) AS `total_amount`, SUM(`USD_value`) AS `USD_value` FROM `ez_requests`";

		if (!empty($where)) {
			$sql .= " WHERE " . implode(' AND ', $where);
		}

		// Execute
		$query = $this->db->query($sql, $params);

		// Return and cache if data found
		if ($query->num_rows() > 0) {
			$result = $query->row();
			$this->cache->save($cache_key, $result, 300); // Cache 5 minutes
			return $result;
		}

		return null;
	}

	
	public function check_valid_completed_address($user_id, $address) {
		$user_id = $this->db->escape_str($user_id);
		$address = $this->db->escape_str($address);
		$sql = "SELECT `request_id` FROM `ez_requests` WHERE `user_id` = ? AND `address` = ? AND `status` = 'Completed' LIMIT 1";
		$query = $this->db->query($sql, array($user_id, $address));
		if($query->num_rows() > 0) {
			return 1;
		}
		return 0;
	}
	
	public function count_unconfirmed_requests($user_id) {
		$user_id = $this->db->escape_str($user_id);
		$sql = "SELECT COUNT(`request_id`) AS cnt FROM `ez_requests` WHERE `user_id` = ? AND `status` = 'Unconfirmed'";
		$query = $this->db->query($sql, array($user_id));
		if($query->num_rows() > 0) {
			return $query->result()[0]->cnt;
		}
		return 0;
	}

	public function get_pending_withdrawal() {
		//$sql = "SELECT `request_id` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`auto_payout`='1' AND `type` LIKE 'Withdrawal' AND `status` = 'Pending' AND `proof` = '' ORDER BY `request_id` ASC LIMIT 1";
		$sql = "SELECT `request_id` FROM `ez_requests`, `ez_users` WHERE `ez_requests`.`user_id` = `ez_users`.`user_id` AND `ez_users`.`banned`='0' AND `ez_users`.`auto_payout`='1' AND `type` LIKE 'Withdrawal' AND `status` = 'Pending' AND `proof` = '' ORDER BY RAND() LIMIT 1";
		$query = $this->db->query($sql);
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_profit_lastweek_by_userid($user_id) {
		$day = date('w');
		if($day == 0) {
			$day = 7;
		}
		$start_time = strtotime('-'.($day-1).' days midnight') - 86400 * 7;
		$end_time = strtotime('+'.(8-$day).' days midnight') - 86400 * 7;
		$sql = "SELECT SUM(IF(STRCMP(`type`, 'deposit') = 0, amount, -amount)) / 100000000 AS profit FROM ez_requests WHERE `user_id` = ? AND status = 'Completed' AND `completion_time` > ? AND `completion_time` < ?";
		$query = $this->db->query($sql, array($user_id, $start_time, $end_time));
		if($query->num_rows() > 0) {
			return $query->result()[0]->profit;
		}
		return 0;
	}
	
	public function get_profit_lastweek_from_referrals($referrer) {
		$day = date('w');
		if($day == 0) {
			$day = 7;
		}
		$start_time = strtotime('-'.($day-1).' days midnight') - 86400 * 7;
		$end_time = strtotime('+'.(8-$day).' days midnight') - 86400 * 7;
		$sql = "SELECT SUM(IF(STRCMP(`type`, 'deposit') = 0, amount, -amount)) / 100000000 AS profit FROM ez_requests WHERE `user_id` IN (SELECT `user_id` FROM `ez_users` WHERE `referrer` = ? ) AND status = 'Completed' AND `completion_time` > ? AND `completion_time` < ?";
		$query = $this->db->query($sql, array($referrer, $start_time, $end_time));
		if($query->num_rows() > 0) {
			return $query->result()[0]->profit;
		}
		return 0;
	}
	
	public function get_confirmed_processing_request($user_id) {
		$sql = "SELECT `request_id` FROM `ez_requests` WHERE `user_id` = ? AND `type` = 'Deposit' AND `status` = 'Processing' AND `confirmations` > 0 ORDER BY `request_id` ASC LIMIT 1";
		$query = $this->db->query($sql, array($user_id));
		if($query->num_rows() > 0) {
			return $query->result()[0]->request_id;
		}
		return 0;
	}
	
	public function get_cashflow_by_range($start, $end, $mode = 'day', $unit = 'BNB') {
        // Determine grouping
        switch ($mode) {
            case 'week':
                $group = "YEAR(date), WEEK(date)";
                $selectLabel = "CONCAT(YEAR(date), '-W', WEEK(date)) AS group_label";
                break;
            case 'month':
                $group = "DATE_FORMAT(date, '%Y-%m')";
                $selectLabel = "DATE_FORMAT(date, '%Y-%m') AS group_label";
                break;
            default:
                $group = "DATE(date)";
                $selectLabel = "DATE(date) AS group_label";
        }
		
		if($unit == 'BNB') {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN type = 'deposit' AND status = 'Completed' THEN amount ELSE 0 END) / 100000000 AS deposit,
					SUM(CASE WHEN type = 'withdrawal' AND status = 'Completed' THEN amount ELSE 0 END) / 100000000 AS withdrawal
				FROM ez_requests
				WHERE status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND DATE(date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(date)
			";
		} else {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN type = 'deposit' AND status = 'Completed' THEN USD_value ELSE 0 END) AS deposit,
					SUM(CASE WHEN type = 'withdrawal' AND status = 'Completed' THEN USD_value ELSE 0 END) AS withdrawal
				FROM ez_requests
				WHERE status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND DATE(date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(date)
			";
		}
        

        return $this->db->query($sql, [$start, $end])->result();
    }
	
	public function get_cashflow_from_user_by_range($user_id, $start, $end, $mode = 'day', $unit = 'BNB') {
		// Determine grouping
		switch ($mode) {
			case 'week':
				$group = "YEAR(ez_requests.date), WEEK(ez_requests.date)";
				$selectLabel = "CONCAT(YEAR(ez_requests.date), '-W', WEEK(ez_requests.date)) AS group_label";
				break;
			case 'month':
				$group = "DATE_FORMAT(ez_requests.date, '%Y-%m')";
				$selectLabel = "DATE_FORMAT(ez_requests.date, '%Y-%m') AS group_label";
				break;
			default:
				$group = "DATE(ez_requests.date)";
				$selectLabel = "DATE(ez_requests.date) AS group_label";
		}

		if ($unit == 'BNB') {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS withdrawal
				FROM ez_requests
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_requests.user_id = ?
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		} else {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS withdrawal
				FROM ez_requests
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_requests.user_id = ?
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		}

		return $this->db->query($sql, [$user_id, $start, $end])->result();
	}
	
	public function get_cashflow_from_referrer_by_range($referrer, $start, $end, $mode = 'day', $unit = 'BNB') {
		// Determine grouping
		switch ($mode) {
			case 'week':
				$group = "YEAR(ez_requests.date), WEEK(ez_requests.date)";
				$selectLabel = "CONCAT(YEAR(ez_requests.date), '-W', WEEK(ez_requests.date)) AS group_label";
				break;
			case 'month':
				$group = "DATE_FORMAT(ez_requests.date, '%Y-%m')";
				$selectLabel = "DATE_FORMAT(ez_requests.date, '%Y-%m') AS group_label";
				break;
			default:
				$group = "DATE(ez_requests.date)";
				$selectLabel = "DATE(ez_requests.date) AS group_label";
		}

		if ($unit == 'BNB') {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS withdrawal
				FROM ez_requests
				INNER JOIN ez_users ON ez_requests.user_id = ez_users.user_id
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_users.referrer = ?
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		} else {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS withdrawal
				FROM ez_requests
				INNER JOIN ez_users ON ez_requests.user_id = ez_users.user_id
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_users.referrer = ?
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		}

		return $this->db->query($sql, [$referrer, $start, $end])->result();
	}
	
	//group feature
	public function get_cashflow_from_group_by_range($group_id, $start, $end, $mode = 'day', $unit = 'BNB') {
		// Determine grouping
		switch ($mode) {
			case 'week':
				$group = "YEAR(ez_requests.date), WEEK(ez_requests.date)";
				$selectLabel = "CONCAT(YEAR(ez_requests.date), '-W', WEEK(ez_requests.date)) AS group_label";
				break;
			case 'month':
				$group = "DATE_FORMAT(ez_requests.date, '%Y-%m')";
				$selectLabel = "DATE_FORMAT(ez_requests.date, '%Y-%m') AS group_label";
				break;
			default:
				$group = "DATE(ez_requests.date)";
				$selectLabel = "DATE(ez_requests.date) AS group_label";
		}

		if ($unit === 'BNB') {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.amount ELSE 0 END) / 100000000 AS withdrawal
				FROM ez_requests
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_requests.user_id IN (
					  SELECT user_id FROM ez_user_groups WHERE group_id = ?
				  )
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		} else {
			$sql = "
				SELECT
					$selectLabel,
					SUM(CASE WHEN ez_requests.type = 'deposit' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS deposit,
					SUM(CASE WHEN ez_requests.type = 'withdrawal' AND ez_requests.status = 'Completed' THEN ez_requests.USD_value ELSE 0 END) AS withdrawal
				FROM ez_requests
				WHERE ez_requests.status = 'Completed'
				  AND CHAR_LENGTH(proof) > 30
				  AND ez_requests.user_id IN (
					  SELECT user_id FROM ez_user_groups WHERE group_id = ?
				  )
				  AND DATE(ez_requests.date) BETWEEN ? AND ?
				GROUP BY $group
				ORDER BY MIN(ez_requests.date)
			";
		}

		return $this->db->query($sql, [$group_id, $start, $end])->result();
	}
	//group feature
	
	public function get_user_request_summary($user_id) {
		$this->db->select("
			SUM(CASE WHEN status = 'Completed' AND type = 'Deposit' THEN amount ELSE 0 END) AS total_deposits,
			SUM(CASE WHEN status = 'Completed' AND type = 'Withdrawal' THEN amount ELSE 0 END) AS total_withdrawals,
			SUM(CASE WHEN type = 'Withdrawal' AND status IN ('Pending', 'Processing', 'Unconfirmed') THEN amount ELSE 0 END) AS pending_balance
		");
		$this->db->from('ez_requests');
		$this->db->where('user_id', $user_id);

		$query = $this->db->get();
		return $query->num_rows() > 0 ? $query->row() : null;
	}
	
	//sync_data
	public function auto_sync_request_data($request_id) {
		if(!SYNC_DATA_ALLOWED) {
			return;
		}
		$request_info = $this->get_request_info_by_id($request_id);
		$apiUrl = 'http://'.SYNC_DATA_IP.'/backup/bnbpick/requests.php';
		$requestData = $request_info;
		$jsonData = json_encode($requestData);
		$cmd = "curl -X POST -H 'Content-Type: application/json' -d '$jsonData' '$apiUrl' > /dev/null 2>&1 &";
		exec($cmd);
	}
	//sync_data
	
}
?>