<?php
class GameModel extends CI_Model {
	
	public $limit_entries = 20;
	public $enable_batch_insert_game = false;
	public $game_cache_queue = array();
	public $cache_limit = 50; 
	
	function __construct() {
		parent::__construct();
		$this->load->driver('cache', ['adapter' => 'memcached', 'backup' => 'file']);
		$this->load->database();
	}
	
	public function create_new_game($user_id, $user_name, $ip, $game_title, $nonce, $client_seed, $server_seed, $server_seed_hash, $bet_amount, $payout = 0, $prediction = '', $data = '', $win_amount = 0, $created_time = 0, $completion_time = 0, $show_server_seed_hash = 0) {
		$game_data = array(
			'user_id' 		=> $user_id,
			'user_name' 	=> $user_name,
			'ip'			=> $ip,
			'game_title' 	=> $game_title,
			'nonce' 		=> $nonce,
			'client_seed' 	=> $client_seed,
			'server_seed'	=> $server_seed,
			'server_seed_hash'	=> $server_seed_hash,
			'bet_amount' 	=> $bet_amount,
			'payout' 		=> $payout,
			'prediction' 	=> $prediction,
			'data' 			=> $data,
			'win_amount' 	=> $win_amount,
			'created_time' 	=> $created_time,
			'completion_time' 	=> $completion_time,
			'show_server_seed_hash' 	=> $show_server_seed_hash,
		);
		
		if($this->enable_batch_insert_game
		&& ($game_title == 'Dice'
		|| $game_title == 'Ultimate'
		|| $game_title == 'Limbo'
		)) {
			$this->game_cache_queue = $this->cache->get('game_cache_queue') ?: [];
			$this->game_cache_queue[] = $game_data;
			$this->cache->save('game_cache_queue', $this->game_cache_queue, 3600);
			if (count($this->game_cache_queue) >= $this->cache_limit) {
				$this->db->insert_batch('ez_games', $this->game_cache_queue);
				$insertedRows = $this->db->affected_rows();
				$this->cache->save('game_cache_queue', [], 3600);
			}
			 
			return -1;
		} else {
			$this->db->insert('ez_games', $game_data);
			$insert_id = $this->db->insert_id();
			return  $insert_id;
		}
	}
	
	public function generate_game_dice_number($nonce, $client_seed, $server_seed) {
		$hash = hash_hmac('sha256', $client_seed.':'.$nonce, $server_seed, true);
		$number = intval(((ord($hash[0]) / 256) + (ord($hash[1]) / (256 * 256)) + (ord($hash[2]) / (256 * 256 * 256)) + (ord($hash[3]) / (256 * 256 * 256 * 256))) * 10000);
		//return number_format($number / 100, 2, '.', '');
		return round($number / 100, 2);	
	}

	public function generate_mines_map($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$hash3 = hash_hmac('sha256', $client_seed.':'.$nonce.':2', $server_seed, true);
		$arr_decs = array();
		$arr_nums = array();
		$arr_mines = array();
		for($i = 0; $i < strlen($hash1); $i++){
		   array_push($arr_decs, ord($hash1[$i]));
		}
		for($i = 0; $i < strlen($hash2); $i++){
		   array_push($arr_decs, ord($hash2[$i]));
		}
		for($i = 0; $i < strlen($hash3); $i++){
		   array_push($arr_decs, ord($hash3[$i]));
		}
		for($i = 0, $j = 25; $i < sizeof($arr_decs), $j > 1; $i+=4, $j--) {
			$number =  intval(($arr_decs[$i] / pow(256, 1)
					+  $arr_decs[$i+1] / pow(256, 2)
					+  $arr_decs[$i+2] / pow(256, 3)
					+  $arr_decs[$i+3] / pow(256, 4)) * $j);
			array_push($arr_nums, $number);
		}
		for($i = 0; $i < sizeof($arr_nums); $i++){
			$number = $arr_nums[$i];
			for($j = 0; $j <= $number; $j++) {
				if(in_array($j, $arr_mines)) {
					$number++;
				}
			}
			array_push($arr_mines, $number);
		}
		return $arr_mines;
	}
	
	public function generate_gems_map($difficulty, $nonce, $client_seed, $server_seed) {
		$gems = array();
		$hashs = array();
		$nums = array();
		$num_rows = 10;
		for($i = 0; $i < $num_rows; $i++) {
			array_push($hashs, hash_hmac('sha256', $client_seed.':'.$nonce.':'.$i, $server_seed, true));
		}
		if($difficulty == 'easy') {
			for($i = 0; $i < $num_rows; $i++) {
				$num = intval(((ord($hashs[$i][0]) / 256) + (ord($hashs[$i][1]) / (256 * 256)) + (ord($hashs[$i][2]) / (256 * 256 * 256)) + (ord($hashs[$i][3]) / (256 * 256 * 256 * 256))) * 3); 
				array_push($nums, $num);
			}
			for($i = 0; $i < $num_rows; $i++) {
				$row = array(0, 0, 0);
				for($j = 0; $j < 3; $j++) {
					if($j != $nums[$i]) {
						$row[$j] = 1;
					}
				}
				array_push($gems, $row);
			}
		} else if($difficulty == 'medium') {
			for($i = 0; $i < $num_rows; $i++) {
				$num = intval(((ord($hashs[$i][0]) / 256) + (ord($hashs[$i][1]) / (256 * 256)) + (ord($hashs[$i][2]) / (256 * 256 * 256)) + (ord($hashs[$i][3]) / (256 * 256 * 256 * 256))) * 2); 
				array_push($nums, $num);
			}
			for($i = 0; $i < $num_rows; $i++) {
				$row = array(0, 0);
				for($j = 0; $j < 2; $j++) {
					if($j == $nums[$i]) {
						$row[$j] = 1;
					}
				}
				array_push($gems, $row);
			}
		} else if($difficulty == 'hard') {
			for($i = 0; $i < $num_rows; $i++) {
				$num = intval(((ord($hashs[$i][0]) / 256) + (ord($hashs[$i][1]) / (256 * 256)) + (ord($hashs[$i][2]) / (256 * 256 * 256)) + (ord($hashs[$i][3]) / (256 * 256 * 256 * 256))) * 3); 
				array_push($nums, $num);
			}
			for($i = 0; $i < $num_rows; $i++) {
				$row = array(0, 0, 0);
				for($j = 0; $j < 3; $j++) {
					if($j == $nums[$i]) {
						$row[$j] = 1;
					}
				}
				array_push($gems, $row);
			}
		}
		return $gems;
	}
	
	/*public function calculate_game_gems_payout($difficulty, $steps) {
		$payout = 1;
		$MAGIC_NUMBER = 0.97;
		if($difficulty == 'easy') {
			for($i = 0; $i < $steps; $i++) {
				$payout = number_format($payout * $MAGIC_NUMBER * 1.5, 2, '.', '');
			}
		}
		if($difficulty == 'medium') {
			for($i = 0; $i < $steps; $i++) {
				$payout = number_format($payout * $MAGIC_NUMBER * 2, 2, '.', '');
			}
		}
		if($difficulty == 'hard') {
			for($i = 0; $i < $steps; $i++) {
				$payout = number_format($payout * $MAGIC_NUMBER * 3, 2, '.', '');
			}
		}
		return $payout;
	}*/
	
	public function calculate_game_gems_payout($difficulty, $steps) {
		$payout = 1.0;
		$MAGIC_NUMBER = 0.97;

		$multipliers = [
			'easy'   => 1.5,
			'medium' => 2,
			'hard'   => 3,
		];

		if (isset($multipliers[$difficulty])) {
			$factor = $MAGIC_NUMBER * $multipliers[$difficulty];
			for ($i = 0; $i < $steps; $i++) {
				$payout = round($payout * $factor, 2);
			}
		}

		return $payout; // float
	}
	
	public function generate_game_slots_data($nonce, $client_seed, $server_seed) {
		$hash = hash_hmac('sha256', $client_seed.':'.$nonce, $server_seed, true);
		$num1 = intval(((ord($hash[0]) / 256) + (ord($hash[1]) / (256 * 256)) + (ord($hash[2]) / (256 * 256 * 256)) + (ord($hash[3]) / (256 * 256 * 256 * 256))) * 36); 
		$num2 = intval(((ord($hash[4]) / 256) + (ord($hash[5]) / (256 * 256)) + (ord($hash[6]) / (256 * 256 * 256)) + (ord($hash[7]) / (256 * 256 * 256 * 256))) * 36); 
		$num3 = intval(((ord($hash[8]) / 256) + (ord($hash[9]) / (256 * 256)) + (ord($hash[10]) / (256 * 256 * 256)) + (ord($hash[11]) / (256 * 256 * 256 * 256))) * 36); 
		return array($num1, $num2, $num3);
	}
	
	public function get_game_slots_reels() {
		$reel1 = array('bell', 'bell', 'orange', 'watermelon', 'watermelon', 'orange', 'watermelon', 'bell', 'bell', 'bar', 'watermelon', 'bell', 'seven', 'bell', 'watermelon', 'bell', 'cherry', 'cherry', 'bell', 'bell', 'bell', 'bell', 'bell', 'watermelon', 'watermelon', 'cherry', 'seven', 'watermelon', 'cherry', 'bell', 'bell', 'watermelon', 'bell', 'cherry', 'bell', 'orange');
		$reel2 = array('orange', 'bell', 'watermelon', 'bell', 'bell', 'bell', 'bell', 'bell', 'cherry', 'watermelon', 'watermelon', 'bell', 'bell', 'bell', 'bell', 'bell', 'cherry', 'orange', 'cherry', 'watermelon', 'watermelon', 'bell', 'watermelon', 'watermelon', 'seven', 'orange', 'cherry', 'bar', 'bell', 'cherry', 'bell', 'cherry', 'bell', 'watermelon', 'watermelon', 'seven');
		$reel3 = array('bell', 'bell', 'bell', 'bell', 'cherry', 'seven', 'watermelon', 'bell', 'watermelon', 'bell', 'bell', 'watermelon', 'bell', 'bell', 'watermelon', 'cherry', 'seven', 'cherry', 'cherry', 'bell', 'bell', 'watermelon', 'orange', 'bell', 'bell', 'watermelon', 'cherry', 'watermelon', 'bell', 'bar', 'watermelon', 'bell', 'orange', 'bell', 'seven', 'watermelon');
		return array($reel1, $reel2, $reel3);
	}
	
	public function calculate_game_slots_payout_by_line($line) {
		$payout = 0;
		$payout_table = array(
			array(array('bar', 'bar', 'bar'), 1000),
			array(array('seven', 'seven', 'seven'), 100),
			array(array('orange', 'orange', 'orange'), 50),
			array(array('cherry', 'cherry', 'cherry'), 15),
			array(array('seven', 'seven', 'orange'), 10),
			array(array('seven', 'seven', 'cherry'), 10),
			array(array('seven', 'seven', 'watermelon'), 10),
			array(array('seven', 'seven', 'bell'), 10),
			array(array('watermelon', 'watermelon', 'watermelon'), 5),
			array(array('bell', 'bell', 'bell'), 2),
			array(array('seven', 'orange', 'orange'), 1),
			array(array('seven', 'orange', 'cherry'), 1),
			array(array('seven', 'orange', 'watermelon'), 1),
			array(array('seven', 'orange', 'bell'), 1),
			array(array('seven', 'cherry', 'cherry'), 1),
			array(array('seven', 'cherry', 'watermelon'), 1),
			array(array('seven', 'cherry', 'bell'), 1),
			array(array('seven', 'watermelon', 'watermelon'), 1),
			array(array('seven', 'watermelon', 'bell'), 1),
			array(array('seven', 'bell', 'bell'), 1),
		);
		for($i = 0; $i < sizeof($payout_table); $i++) {
			if($this->match_game_slots_line($line, $payout_table[$i][0])) {
				$payout = $payout_table[$i][1];
				break;
			}
		}
		return $payout;
	}
	
	public function match_game_slots_line($line, $pattern) {
		$matching_count = 0;
		for($i = 0; $i < sizeof($line); $i++) {
			for($j = 0; $j < sizeof($pattern); $j++) {
				if($line[$i] == $pattern[$j]) {
					$matching_count++;
					$pattern[$j] = '';
					break;
				}
			}
		}
		if(array_search('bar', $line) !==  false && array_count_values($line)['bar'] + $matching_count == sizeof($pattern)) {
			return true;
		} else if($matching_count == sizeof($pattern)){
			return true;
		}
		return false;
	}
	
	public function generate_game_poker_data($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$hash3 = hash_hmac('sha256', $client_seed.':'.$nonce.':2', $server_seed, true);
		$hash4 = hash_hmac('sha256', $client_seed.':'.$nonce.':3', $server_seed, true);
		$hash5 = hash_hmac('sha256', $client_seed.':'.$nonce.':4', $server_seed, true);
		$hash6 = hash_hmac('sha256', $client_seed.':'.$nonce.':5', $server_seed, true);
		$hash7 = hash_hmac('sha256', $client_seed.':'.$nonce.':6', $server_seed, true);
		$arr_decs = array();
		$arr_nums = array();
		$arr_cards = array();
		for($i = 0; $i < strlen($hash1); $i++){
		   array_push($arr_decs, ord($hash1[$i]));
		}
		for($i = 0; $i < strlen($hash2); $i++){
		   array_push($arr_decs, ord($hash2[$i]));
		}
		for($i = 0; $i < strlen($hash3); $i++){
		   array_push($arr_decs, ord($hash3[$i]));
		}
		for($i = 0; $i < strlen($hash4); $i++){
		   array_push($arr_decs, ord($hash4[$i]));
		}
		for($i = 0; $i < strlen($hash5); $i++){
		   array_push($arr_decs, ord($hash5[$i]));
		}
		for($i = 0; $i < strlen($hash6); $i++){
		   array_push($arr_decs, ord($hash6[$i]));
		}
		for($i = 0; $i < strlen($hash7); $i++){
		   array_push($arr_decs, ord($hash7[$i]));
		}
		for($i = 0, $j = 52; $i < sizeof($arr_decs), $j >= 1; $i+=4, $j--) {
			$number =  intval(($arr_decs[$i] / pow(256, 1)
					+  $arr_decs[$i+1] / pow(256, 2)
					+  $arr_decs[$i+2] / pow(256, 3)
					+  $arr_decs[$i+3] / pow(256, 4)) * $j);
			array_push($arr_nums, $number);
		}
		for($i = 0; $i < sizeof($arr_nums); $i++){
			$number = $arr_nums[$i];
			for($j = 0; $j <= $number; $j++) {
				if(in_array($j, $arr_cards)) {
					$number++;
				}
			}
			array_push($arr_cards, $number);
		}
		return array_slice($arr_cards, 0, 10);
	}
	
	public function calculate_game_poker_result($cards) {
		if(sizeof($cards) != 5) {
			return null;
		}
		sort($cards);
		
		if(($cards[0] == 32 && $cards[1] == 36 && $cards[2] == 40 && $cards[3] == 44 && $cards[4] == 48)
			||($cards[0] == 33 && $cards[1] == 37 && $cards[2] == 41 && $cards[3] == 45 && $cards[4] == 49)
			||($cards[0] == 34 && $cards[1] == 38 && $cards[2] == 42 && $cards[3] == 46 && $cards[4] == 50)
			||($cards[0] == 35 && $cards[1] == 39 && $cards[2] == 43 && $cards[3] == 47 && $cards[4] == 51)) {
			return array('royal_flush', array($cards[0], $cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if(($cards[1] - $cards[0] == 4 &&  $cards[2] - $cards[1] == 4 &&  $cards[3] - $cards[2] == 4 &&  $cards[4] - $cards[3] == 4)
			||($cards[1] - $cards[0] == 4 &&  $cards[2] - $cards[1] == 4 &&  $cards[3] - $cards[2] == 4 &&  $cards[4] - $cards[3] == 36)) {
			return array('straight_flush', array($cards[0], $cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if(floor($cards[0] / 4) == floor($cards[1] / 4)
			&& floor($cards[1] / 4) == floor($cards[2] / 4)
			&& floor($cards[2] / 4)== floor($cards[3] / 4)){
			return array('four_of_a_kind', array($cards[0], $cards[1], $cards[2], $cards[3]));
		}
		
		if(floor($cards[1] / 4) == floor($cards[2] / 4)
			&&  floor($cards[2] / 4) == floor($cards[3] / 4)
			&&  floor($cards[3] / 4)== floor($cards[4] / 4)) {
			return array('four_of_a_kind', array($cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if((floor($cards[0] / 4) == floor($cards[1] / 4)
			&&  floor($cards[1] / 4) == floor($cards[2] / 4)
			&&  floor($cards[3] / 4)== floor($cards[4] / 4))
		|| (floor($cards[0] / 4) == floor($cards[1] / 4)
			&&  floor($cards[2] / 4) == floor($cards[3] / 4)
			&&  floor($cards[3] / 4)== floor($cards[4] / 4))) {
			return array('full_house', array($cards[0], $cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if($cards[0] % 4 == $cards[1] % 4
			&& $cards[1] % 4 == $cards[2] % 4
			&& $cards[2] % 4 == $cards[3] % 4
			&& $cards[3] % 4 == $cards[4] % 4
		) {
			return array('flush', array($cards[0], $cards[1], $cards[2], $cards[3], $cards[4]));
		}
	
		if((floor($cards[1] / 4) - floor($cards[0] / 4) == 1
			&& floor($cards[2] / 4) - floor($cards[1] / 4) == 1
			&& floor($cards[3] / 4) - floor($cards[2] / 4) == 1
			&& floor($cards[4] / 4) - floor($cards[3] / 4) == 1) 
		|| (floor($cards[1] / 4) - floor($cards[0] / 4) == 1
			&& floor($cards[2] / 4) - floor($cards[1] / 4) == 1
			&& floor($cards[3] / 4) - floor($cards[2] / 4) == 1
			&& floor($cards[4] / 4) - floor($cards[3] / 4) == 9)) {
			return array('straight', array($cards[0], $cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if(floor($cards[0] / 4) == floor($cards[1] / 4) &&  floor($cards[1] / 4) == floor($cards[2] / 4)
		) {
			return array('three_of_a_kind', array($cards[0], $cards[1], $cards[2]));
		}
		
		if(floor($cards[1] / 4) == floor($cards[2] / 4) &&  floor($cards[2] / 4)== floor($cards[3] / 4)
		) {
			return array('three_of_a_kind', array($cards[1], $cards[2], $cards[3]));
		}
		
		if(floor($cards[2] / 4) == floor($cards[3] / 4) &&  floor($cards[3] / 4)== floor($cards[4] / 4)) {
			return array('three_of_a_kind', array($cards[2], $cards[3], $cards[4]));
		}
		
		if(floor($cards[1] / 4) == floor($cards[2] / 4) &&  floor($cards[3] / 4) == floor($cards[4] / 4)) {
			return array('two_pair', array($cards[1], $cards[2], $cards[3], $cards[4]));
		}
		
		if(floor($cards[0] / 4) == floor($cards[1] / 4) &&  floor($cards[3] / 4)== floor($cards[4] / 4)) {
			return array('two_pair', array($cards[0], $cards[1], $cards[3], $cards[4]));
		}
		
		if(floor($cards[0] / 4) == floor($cards[1] / 4) &&  floor($cards[2] / 4)== floor($cards[3] / 4)) {
			return array('two_pair', array($cards[0], $cards[1], $cards[2], $cards[3]));
		}
		
		if(floor($cards[0] / 4) == floor($cards[1] / 4) &&  floor($cards[0] / 4) >= 9) {
			return array('pair_of_jack', array($cards[0], $cards[1]));
		}
		
		if(	floor($cards[1] / 4) == floor($cards[2] / 4) &&  floor($cards[1] / 4) >= 9) {
			return array('pair_of_jack', array($cards[1], $cards[2]));
		}
		
		if(	floor($cards[2] / 4) == floor($cards[3] / 4) &&  floor($cards[2] / 4) >= 9) {
			return array('pair_of_jack', array($cards[2], $cards[3]));
		}
		
		if(	floor($cards[3] / 4) == floor($cards[4] / 4) &&  floor($cards[3] / 4) >= 9) {
			return array('pair_of_jack', array($cards[3], $cards[4]));
		}
			
		return null;
	}
	
	public function calculate_game_poker_max_payout($cards) {
		$max_payout = 0;
		for($m = 0; $m <=1; $m++) {
			for($n = 0; $n <=1; $n++) {
				for($j = 0; $j <=1; $j++) {
					for($q = 0; $q <=1; $q++) {
						for($k = 0; $k <=1; $k++) {
							$poker_result = $this->calculate_game_poker_result(array($cards[$m*5], $cards[$n*5+1], $cards[$j*5+2], $cards[$q*5+3], $cards[$k*5+4]));
							if($poker_result == null) {
								//$max_payout = 0;
							} else {
								$poker_result = $poker_result[0];
								switch($poker_result) {
									case 'royal_flush':
									$max_payout = 800;
									break;
									case 'straight_flush':
									$max_payout = max($max_payout, 60);
									break;
									case 'four_of_a_kind':
									$max_payout = max($max_payout, 22);
									break;
									case 'full_house':
									$max_payout = max($max_payout, 9);
									break;
									case 'flush':
									$max_payout = max($max_payout, 6);
									break;
									case 'straight':
									$max_payout = max($max_payout, 4);
									break;
									case 'three_of_a_kind':
									$max_payout = max($max_payout, 3);
									break;
									case 'two_pair':
									$max_payout = max($max_payout, 2);
									break;
									case 'pair_of_jack':
									$max_payout = max($max_payout, 1);
									break;
								}
							}
						}
					}
				}
			}
		}
		return $max_payout;
	}
	
	
	public function generate_game_baccarat_data($nonce, $client_seed, $server_seed) {
		$hash = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$card1 = intval(((ord($hash[0]) / 256) + (ord($hash[1]) / (256 * 256)) + (ord($hash[2]) / (256 * 256 * 256)) + (ord($hash[3]) / (256 * 256 * 256 * 256))) * 52);
		$card2 = intval(((ord($hash[4]) / 256) + (ord($hash[5]) / (256 * 256)) + (ord($hash[6]) / (256 * 256 * 256)) + (ord($hash[7]) / (256 * 256 * 256 * 256))) * 52);
		$card3 = intval(((ord($hash[8]) / 256) + (ord($hash[9]) / (256 * 256)) + (ord($hash[10]) / (256 * 256 * 256)) + (ord($hash[11]) / (256 * 256 * 256 * 256))) * 52);
		$card4 = intval(((ord($hash[12]) / 256) + (ord($hash[13]) / (256 * 256)) + (ord($hash[14]) / (256 * 256 * 256)) + (ord($hash[15]) / (256 * 256 * 256 * 256))) * 52);
		$card5 = intval(((ord($hash[16]) / 256) + (ord($hash[17]) / (256 * 256)) + (ord($hash[18]) / (256 * 256 * 256)) + (ord($hash[19]) / (256 * 256 * 256 * 256))) * 52);
		$card6 = intval(((ord($hash[20]) / 256) + (ord($hash[21]) / (256 * 256)) + (ord($hash[22]) / (256 * 256 * 256)) + (ord($hash[23]) / (256 * 256 * 256 * 256))) * 52);
		return array($card1, $card2, $card3, $card4, $card5, $card6);
	}
	
	public function calculate_game_baccarat_point($card_value) {
		$number = 0;
		if(floor($card_value / 4) < 8) {
			$number = floor($card_value / 4) + 2;
		} else if (floor($card_value / 4) < 12) {
			$number = 0;
		} else {
			$number = 1;
		}
		return $number;
	}
	
	public function calculate_game_baccarat_result($cards) {
		$player_cards = array($cards[0], $cards[1]);
		$banker_cards = array($cards[2], $cards[3]);
		$player_total = ($this->calculate_game_baccarat_point($player_cards[0]) + $this->calculate_game_baccarat_point($player_cards[1])) % 10;
		$banker_total = ($this->calculate_game_baccarat_point($banker_cards[0]) + $this->calculate_game_baccarat_point($banker_cards[1])) % 10;
		
		if($player_total < 6 && $banker_total < 8) {
			array_push($player_cards, $cards[4]);
			$player_total = ($player_total + $this->calculate_game_baccarat_point($player_cards[2])) % 10;
		}
		
		if(count($player_cards) == 2) {
			if($banker_total < 3 && $player_total < 8) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			}
		} else {
			$player_3rd_card = $this->calculate_game_baccarat_point($player_cards[2]);
			if($banker_total < 3) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			} else if($banker_total == 3 && $player_3rd_card != 8) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			} else if($banker_total == 4 && ($player_3rd_card != 0 && $player_3rd_card != 1 && $player_3rd_card != 8 && $player_3rd_card != 9)) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			} else if($banker_total == 5 && ($player_3rd_card == 4 || $player_3rd_card == 5 || $player_3rd_card == 6 || $player_3rd_card == 7)) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			} else if($banker_total == 6 && ($player_3rd_card == 6 || $player_3rd_card == 7)) {
				array_push($banker_cards, $cards[5]);
				$banker_total = ($banker_total + $this->calculate_game_baccarat_point($banker_cards[2])) % 10;
			}
		}
		if($player_total > $banker_total) {
			return array(1, $player_cards, $player_total, $banker_cards, $banker_total);
		} else if ($player_total < $banker_total) {
			return array(2, $player_cards, $player_total, $banker_cards, $banker_total);
		} else {
			return array(3, $player_cards, $player_total, $banker_cards, $banker_total);
		}
	}
	
	public function generate_game_hilow_data($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$hash3 = hash_hmac('sha256', $client_seed.':'.$nonce.':2', $server_seed, true);
		$hash4 = hash_hmac('sha256', $client_seed.':'.$nonce.':3', $server_seed, true);
		$hash5 = hash_hmac('sha256', $client_seed.':'.$nonce.':4', $server_seed, true);
		$hash6 = hash_hmac('sha256', $client_seed.':'.$nonce.':5', $server_seed, true);
		$hash7 = hash_hmac('sha256', $client_seed.':'.$nonce.':6', $server_seed, true);
		$arr_decs = array();
		$arr_cards = array();
		for($i = 0; $i < strlen($hash1); $i++){
		   array_push($arr_decs, ord($hash1[$i]));
		}
		for($i = 0; $i < strlen($hash2); $i++){
		   array_push($arr_decs, ord($hash2[$i]));
		}
		for($i = 0; $i < strlen($hash3); $i++){
		   array_push($arr_decs, ord($hash3[$i]));
		}
		for($i = 0; $i < strlen($hash4); $i++){
		   array_push($arr_decs, ord($hash4[$i]));
		}
		for($i = 0; $i < strlen($hash5); $i++){
		   array_push($arr_decs, ord($hash5[$i]));
		}
		for($i = 0; $i < strlen($hash6); $i++){
		   array_push($arr_decs, ord($hash6[$i]));
		}
		for($i = 0; $i < strlen($hash7); $i++){
		   array_push($arr_decs, ord($hash7[$i]));
		}
		for($i = 0; $i < sizeof($arr_decs); $i+=4) {
			$number =  intval(($arr_decs[$i] / pow(256, 1)
					+  $arr_decs[$i+1] / pow(256, 2)
					+  $arr_decs[$i+2] / pow(256, 3)
					+  $arr_decs[$i+3] / pow(256, 4)) * 52);
			array_push($arr_cards, $number);
		}
		return array_slice($arr_cards, 0, 52);
	}
	
	public function calculate_game_hilow_number($card_value) {
		$number = 0;
		$card_value = $card_value % 52;
		if(floor($card_value / 4) <= 11) {
			$number = floor($card_value / 4) + 2;
		} else {
			$number = 1;
		}
		return $number;
	}
	
	/*public function calculate_game_hilow_profit($cards, $selections) {
		$total_profit = 1;
		$profit_higher = 1;
		$profit_lower = 1;
		$MAGIC_NUMBER = 97;
		for($i = 0; $i < count($selections); $i++) {
			$number = $this->calculate_game_hilow_number($cards[$i]);
			//selections: lower(1) | lower or equal(2) | equal(3) | higher or equal(4) | higher(5)
			if($number == 1) {
				$profit_higher = number_format(floatval(round(floatval(13 / 12) * $MAGIC_NUMBER) / 100), 2, '.', ''); //higher
				$profit_lower = number_format(floatval(round(floatval(13 / 1) * $MAGIC_NUMBER) / 100), 2, '.', ''); // equal
				if($selections[$i] == 5) {
					$total_profit *= $profit_higher;
				} else if($selections[$i] == 3) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}
			} else if($number == 13) {
				$profit_higher = number_format(floatval(round(floatval(13 / 1) * $MAGIC_NUMBER) / 100), 2, '.', ''); //equal
				$profit_lower = number_format(floatval(round(floatval(13 / 12) * $MAGIC_NUMBER) / 100), 2, '.', ''); // lower
				if($selections[$i] == 3) {
					$total_profit *= $profit_higher;
				} else if($selections[$i] == 1) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}
			} else {
				$profit_higher = number_format(floatval(round(floatval(13 / (13 - $number + 1)) * $MAGIC_NUMBER) / 100), 2, '.', ''); //higher or equal
				$profit_lower = number_format(floatval(round(floatval(13 / $number) * $MAGIC_NUMBER) / 100), 2, '.', ''); // lower or equal
				if($selections[$i] == 4) {
					$total_profit *= $profit_higher;
				} else if($selections[$i] == 2) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}
			}
		}
		return $total_profit;
	}*/
	
	public function calculate_game_hilow_profit($cards, $selections) {
		$total_profit = 1.0;
		$MAGIC_NUMBER = 0.97;

		for ($i = 0; $i < count($selections); $i++) {
			$number = $this->calculate_game_hilow_number($cards[$i]);
			// selections: lower(1) | lower or equal(2) | equal(3) | higher or equal(4) | higher(5)

			if ($number == 1) {
				$profit_higher = round((13 / 12) * $MAGIC_NUMBER, 2); // higher
				$profit_lower  = round((13 / 1)  * $MAGIC_NUMBER, 2); // equal

				if ($selections[$i] == 5) {
					$total_profit *= $profit_higher;
				} elseif ($selections[$i] == 3) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}

			} elseif ($number == 13) {
				$profit_higher = round((13 / 1)  * $MAGIC_NUMBER, 2); // equal
				$profit_lower  = round((13 / 12) * $MAGIC_NUMBER, 2); // lower

				if ($selections[$i] == 3) {
					$total_profit *= $profit_higher;
				} elseif ($selections[$i] == 1) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}

			} else {
				$profit_higher = round((13 / (13 - $number + 1)) * $MAGIC_NUMBER, 2); // higher or equal
				$profit_lower  = round((13 / $number) * $MAGIC_NUMBER, 2);            // lower or equal

				if ($selections[$i] == 4) {
					$total_profit *= $profit_higher;
				} elseif ($selections[$i] == 2) {
					$total_profit *= $profit_lower;
				} else {
					$total_profit = 0;
				}
			}
		}

		return $total_profit; // float
	}
	
		
	public function generate_game_roulette_number($nonce, $client_seed, $server_seed) {
		$hash = hash_hmac('sha256', $client_seed.':'.$nonce, $server_seed, true);
		$number = intval(((ord($hash[0]) / 256) + (ord($hash[1]) / (256 * 256)) + (ord($hash[2]) / (256 * 256 * 256)) + (ord($hash[3]) / (256 * 256 * 256 * 256))) * 37);
		return $number;	
	}
	
	public function check_valid_game_roulette_chip_bet($chip_bet){
		for($i = 0; $i < count($chip_bet); $i++) {
			if(intval($chip_bet[$i][0]) < 0 || intval($chip_bet[$i][0]) > 156) {
				return false;
			}
		}
		return true;
	}
	
	public function calculate_game_roulette_profit($chip_bet, $number) {
		$bet_places = [
			[0], [1], [2], [3], [4], [5], [6], [7], [8], [9], [10], [11], [12], [13], [14], [15], [16], [17], [18], [19], [20], [21], [22], [23], [24], [25], [26], [27], [28], [29], [30], [31], [32], [33], [34], [35], [36], //0-36
			[1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18],															//37 1-18
			[19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36],												//38 19-39
			[2, 4, 6, 8, 10, 11, 13, 15, 17, 20, 22, 24, 26, 28, 29, 31, 33, 35],									//39 black
			[1, 3, 5, 7, 9, 12, 14, 16, 18, 19, 21, 23, 25, 27, 30, 32, 34, 36],									//40 red
			[2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36],									//41 even
			[1, 3, 5, 7, 9, 11, 13, 15, 17, 19, 21, 23, 25, 27, 29, 31, 33, 35],									//42 odd
			[1,2,3,4,5,6,7,8,9,10,11,12],																			//43 1-12
			[13,14,15,16,17,18,19,20,21,22,23,24],																	//44 13-24
			[25,26,27,28,29,30,31,32,33,34,35,36],																	//45 25-36
			[1, 4, 7, 10, 13, 16, 19, 22, 25, 28, 31, 34],															//46 column1
			[2, 5, 8, 11, 14, 17, 20, 23, 26, 29, 32, 35],															//47 column2
			[3, 6, 9, 12, 15, 18, 21, 24, 27, 30, 33, 36],															//48 column3
			[0, 1],																									//49
			[1, 2],
			[0, 2],
			[2, 3],
			[0, 3],
			[1, 4],
			[4, 5],
			[2, 5],
			[5, 6],
			[3, 6],
			[4, 7],
			[7, 8],
			[5, 8],
			[8, 9],
			[6, 9],
			[7, 10],
			[10, 11],
			[8, 11],
			[11, 12],
			[9, 12],
			[10, 13],
			[13, 14],
			[11, 14],
			[14, 15],
			[12, 15],
			[13, 16],
			[16, 17],
			[14, 17],
			[17, 18],
			[15, 18],
			[16, 19],
			[19, 20],
			[17, 20],
			[20, 21],
			[18, 21],
			[19, 22],
			[22, 23],
			[20, 23],
			[23, 24],
			[21, 24],
			[22, 25],
			[25, 26],
			[23, 26],
			[26, 27],
			[24, 27],
			[25, 28],
			[28, 29],
			[26, 29],
			[29, 30],
			[27, 30],
			[28, 31],
			[31, 32],
			[29, 32],
			[32, 33],
			[-1],
			[30, 33],
			[31, 34],
			[34, 35],
			[32, 35],
			[35, 36],
			[33, 36],																									//109																																	
			[0, 1, 2],																									//110
			[0, 2, 3],
			[1, 2, 3],
			[4, 5, 6],
			[7, 8, 9],
			[10, 11, 12],
			[13, 14, 15],
			[16, 17, 18],
			[19, 20, 21],
			[22, 23, 24],
			[25, 26, 27],
			[28, 29, 30],
			[31, 32, 33],
			[34, 35, 36],
			[1, 2, 4, 5],
			[2, 3, 5, 6],
			[4, 5, 7, 8],
			[5, 6, 8, 9],
			[7, 8, 10, 11],
			[8, 9, 11, 12],
			[10, 11, 13, 14],
			[11, 12, 14, 15],
			[13, 14, 16, 17],
			[14, 15, 17, 18],
			[16, 17, 19, 20],
			[17, 18, 20, 21],
			[19, 20, 22, 23],
			[20, 21, 23, 24],
			[22, 23, 25, 26],
			[23, 24, 26, 27],
			[25, 26, 28, 29],
			[26, 27, 29, 30],
			[28, 29, 31, 32],
			[29, 30, 32, 33],
			[31, 32, 34, 35],
			[32, 33, 35, 36],
			[1, 2, 3, 4, 5, 6],
			[4, 5, 6, 7, 8, 9],
			[7, 8, 9, 10, 11, 12],
			[10, 11, 12, 13, 14, 15],
			[13, 14, 15, 16, 17, 18],
			[16, 17, 18, 19, 20, 21],
			[19, 20, 21, 22, 23, 24],
			[22, 23, 24, 25, 26, 27],
			[25, 26, 27, 28, 29, 30],
			[28, 29, 30, 31, 32, 33],
			[31, 32, 33, 34, 35, 36]																					//156
		];
		$profit = 0;
		for($i = 0; $i < count($chip_bet); $i++) {
			$place_id = intval($chip_bet[$i][0]);
			$bet_amount = (100 * pow(10, intval($chip_bet[$i][1])));
			$multiplier = intval(36 /  count($bet_places[$place_id]));
			for($j = 0; $j < count($bet_places[$place_id]); $j++) {
				if($number == $bet_places[$place_id][$j]) {
					$profit += $multiplier * $bet_amount;
				}
			}
		}
		return $profit;
	}

	public function generate_game_wheel_index($nonce, $client_seed, $server_seed) {
		$hash = hash_hmac('sha256', $client_seed.':'.$nonce, $server_seed, true);
		$number = intval(((ord($hash[0]) / 256) + (ord($hash[1]) / (256 * 256)) + (ord($hash[2]) / (256 * 256 * 256)) + (ord($hash[3]) / (256 * 256 * 256 * 256))) * 12);
		return $number;	
	}
	
	function generate_game_plinko_direction($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$row1 = intval(((ord($hash1[0]) / 256) + (ord($hash1[1]) / (256 * 256)) + (ord($hash1[2]) / (256 * 256 * 256)) + (ord($hash1[3]) / (256 * 256 * 256 * 256))) * 2);
		$row2 = intval(((ord($hash1[4]) / 256) + (ord($hash1[5]) / (256 * 256)) + (ord($hash1[6]) / (256 * 256 * 256)) + (ord($hash1[7]) / (256 * 256 * 256 * 256))) * 2);
		$row3 = intval(((ord($hash1[8]) / 256) + (ord($hash1[9]) / (256 * 256)) + (ord($hash1[10]) / (256 * 256 * 256)) + (ord($hash1[11]) / (256 * 256 * 256 * 256))) * 2);
		$row4 = intval(((ord($hash1[12]) / 256) + (ord($hash1[13]) / (256 * 256)) + (ord($hash1[14]) / (256 * 256 * 256)) + (ord($hash1[15]) / (256 * 256 * 256 * 256))) * 2);
		$row5 = intval(((ord($hash1[16]) / 256) + (ord($hash1[17]) / (256 * 256)) + (ord($hash1[18]) / (256 * 256 * 256)) + (ord($hash1[19]) / (256 * 256 * 256 * 256))) * 2);
		$row6 = intval(((ord($hash1[20]) / 256) + (ord($hash1[21]) / (256 * 256)) + (ord($hash1[22]) / (256 * 256 * 256)) + (ord($hash1[23]) / (256 * 256 * 256 * 256))) * 2);
		$row7 = intval(((ord($hash1[24]) / 256) + (ord($hash1[25]) / (256 * 256)) + (ord($hash1[26]) / (256 * 256 * 256)) + (ord($hash1[27]) / (256 * 256 * 256 * 256))) * 2);
		$row8 = intval(((ord($hash1[28]) / 256) + (ord($hash1[29]) / (256 * 256)) + (ord($hash1[30]) / (256 * 256 * 256)) + (ord($hash1[31]) / (256 * 256 * 256 * 256))) * 2);
		$row9 = intval(((ord($hash2[0]) / 256) + (ord($hash2[1]) / (256 * 256)) + (ord($hash2[2]) / (256 * 256 * 256)) + (ord($hash2[3]) / (256 * 256 * 256 * 256))) * 2);
		$row10 = intval(((ord($hash2[4]) / 256) + (ord($hash2[5]) / (256 * 256)) + (ord($hash2[6]) / (256 * 256 * 256)) + (ord($hash2[7]) / (256 * 256 * 256 * 256))) * 2);
		return $row1*(2**9)+$row2*(2**8)+$row3*(2**7)+$row4*(2**6)+$row5*(2**5)+$row6*(2**4)+$row7*(2**3)+$row8*(2**2)+$row9*2+$row10;	
	}
	
	function calculate_game_plinko_payout($risk, $direction_idx) {
		$bin = strval(decbin($direction_idx));
		$slot_idx = 0;
		for($i = 0; $i < strlen($bin); $i++) {
			$slot_idx += $bin[$i];
		}
		$risk_payouts = array(
			array(9, 2.4, 1.3, 1.1, 1, 0.5, 1, 1.1, 1.3, 2.4, 9),
			array(36, 5, 2, 1.2, 0.6, 0.4, 0.6, 1.2, 2, 5, 36),
			array(75, 10, 3, 0.7, 0.4, 0.15, 0.4, 0.7, 3, 10, 75)
		);
		$payout = $risk_payouts[$risk][$slot_idx];
		return $payout;
	}
	
	/*function generate_game_limbo_number($nonce, $client_seed, $server_seed) {
		$MAGIC_NUMBER = 16777216;
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce, $server_seed, true);
		$number = $MAGIC_NUMBER / (intval(((ord($hash1[0]) / 256) + (ord($hash1[1]) / (256 * 256)) + (ord($hash1[2]) / (256 * 256 * 256)) + (ord($hash1[3]) / (256 * 256 * 256 * 256))) * $MAGIC_NUMBER) + 1) * (1 - 0.03);
		return number_format($number, 2, '.', '');
	}*/
	
	function generate_game_limbo_number($nonce, $client_seed, $server_seed) {
		$MAGIC_NUMBER = 16777216; // 2^24
		$edge = 0.03; // house edge 3%

		$hash1 = hash_hmac('sha256', $client_seed . ':' . $nonce, $server_seed, true);

		// Lấy 4 byte đầu từ hash để làm số ngẫu nhiên
		$r = (
			(ord($hash1[0]) / 256) +
			(ord($hash1[1]) / (256 * 256)) +
			(ord($hash1[2]) / (256 * 256 * 256)) +
			(ord($hash1[3]) / (256 * 256 * 256 * 256))
		);

		// Tính payout (float)
		$number = $MAGIC_NUMBER / (($r * $MAGIC_NUMBER) + 1) * (1 - $edge);

		return round($number, 2); // float, làm tròn 2 số thập phân
	}
	
	public function generate_game_keno_numbers($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$arr_decs = array();
		$arr_nums = array();
		$arr_keno_nums = array();
		for($i = 0; $i < strlen($hash1); $i++){
		   array_push($arr_decs, ord($hash1[$i]));
		}
		for($i = 0; $i < strlen($hash2); $i++){
		   array_push($arr_decs, ord($hash2[$i]));
		}
		for($i = 0, $j = 40; $i < sizeof($arr_decs), $j > 30; $i+=4, $j--) {
			$number =  intval(($arr_decs[$i] / pow(256, 1)
					+  $arr_decs[$i+1] / pow(256, 2)
					+  $arr_decs[$i+2] / pow(256, 3)
					+  $arr_decs[$i+3] / pow(256, 4)) * $j);
			array_push($arr_nums, $number);
		}
		for($i = 0; $i < sizeof($arr_nums); $i++){
			$number = $arr_nums[$i];
			for($j = 0; $j <= $number; $j++) {
				if(in_array($j, $arr_keno_nums)) {
					$number++;
				}
			}
			array_push($arr_keno_nums, $number);
		}
		for($i = 0; $i < sizeof($arr_keno_nums); $i++) {
			$arr_keno_nums[$i]++;
		}
		return $arr_keno_nums;
	}
	
	public function check_valid_selected_numbers_game_keno($numbers) {
		if(!is_array($numbers)) {
			return false;
		}
		if(count($numbers) < 1 || count($numbers) > 10) {
			return false;
		}
		if(count($numbers) !== count(array_unique($numbers))) {
			return false;
		}
		for($i = 0; $i < count($numbers); $i++) {
			$numbers[$i] = intval($numbers[$i]);
			if($numbers[$i] < 1 || $numbers[$i] > 40) {
				return false;
			}
		}
		return true;
	}
	
	public function calculate_game_keno_multiplier($arr1, $arr2, $difficulty) {
		$matched_count = 0;
		$keno_multipliers = array(
			array(
				array(0, 3.88),
				array(0, 1.86, 4.4),
				array(0, 1, 3, 9.8),
				array(0, 0.8, 1.71, 5, 22),
				array(0, 0.25, 1.3, 4.3, 15.7, 35),
				array(0, 0, 1, 3.6, 6, 23, 50),
				array(0, 0, 0.5, 2.2, 6, 21, 34, 60),
				array(0, 0, 0, 2, 4, 15, 20, 55, 70),
				array(0, 0, 0, 1.5, 3, 7.5, 17, 40, 60, 86),
				array(0, 0, 0, 1.3, 2.3, 4.5, 7.9, 20, 50, 80, 100)
			),
			array(
				array(0.7, 1.78),
				array(0, 2, 3.48),
				array(0, 1, 2.1, 20),
				array(0, 0, 2, 9, 81),
				array(0, 0, 1, 3, 35, 310),
				array(0, 0, 1, 2, 10, 60, 700),
				array(0, 0, 1.1, 1.3, 4, 25, 80, 750),
				array(0, 0, 1.1, 1.3, 1.8, 10, 20, 135, 800),
				array(0, 0, 1.1, 1.3, 1.6, 2.5, 5, 40, 200, 900),
				array(0, 0, 1.1, 1.2, 1.3, 1.5, 2.7, 12, 100, 250, 1000)
			),
			array(
				array(0.4, 2.68),
				array(0, 1.8, 4.8),
				array(0, 0, 3.1, 45),
				array(0, 0, 1.8, 9, 100),
				array(0, 0, 1.4, 4, 14, 339),
				array(0, 0, 0, 3, 9, 170, 730),
				array(0, 0, 0, 2, 7, 30, 340, 800),
				array(0, 0, 0, 2, 4, 10, 62, 400, 900),
				array(0, 0, 0, 2, 2.4, 5, 12, 100, 500, 1000),
				array(0, 0, 0, 1.6, 2, 3.5, 7.2, 26, 100, 500, 1000)
			),
			array(
				array(0, 3.88),
				array(0, 0, 16.8),
				array(0, 0, 0, 79.8),
				array(0, 0, 0, 10, 250),
				array(0, 0, 0, 4.5, 45, 475),
				array(0, 0, 0, 0, 10, 350, 770),
				array(0, 0, 0, 0, 7, 84, 445, 800),
				array(0, 0, 0, 0, 5, 20, 250, 680, 900),
				array(0, 0, 0, 0, 4, 11, 55, 410, 800, 1000),
				array(0, 0, 0, 0, 3.5, 8, 11, 50, 500, 800, 1000)
			)
		);
		for($i = 0; $i < count($arr1); $i++) {
			for($j = 0; $j < count($arr2); $j++) {
				if($arr1[$i] == $arr2[$j]) {
					$matched_count++;
				}
			}
		}
		if($difficulty == 'easy') {
			return $keno_multipliers[0][count($arr2)-1][$matched_count];
		}
		if($difficulty == 'medium') {
			return $keno_multipliers[1][count($arr2)-1][$matched_count];
		}
		if($difficulty == 'hard') {
			return $keno_multipliers[2][count($arr2)-1][$matched_count];
		}
		if($difficulty == 'extreme') {
			return $keno_multipliers[3][count($arr2)-1][$matched_count];
		}
	}
	
	public function generate_game_triple_dice($nonce, $client_seed, $server_seed) {
		$hash1 = hash_hmac('sha256', $client_seed.':'.$nonce.':0', $server_seed, true);
		$hash2 = hash_hmac('sha256', $client_seed.':'.$nonce.':1', $server_seed, true);
		$hash3 = hash_hmac('sha256', $client_seed.':'.$nonce.':2', $server_seed, true);
		$number1 = intval(((ord($hash1[0]) / 256) + (ord($hash1[1]) / (256 * 256)) + (ord($hash1[2]) / (256 * 256 * 256)) + (ord($hash1[3]) / (256 * 256 * 256 * 256))) * 6) + 1;
		$number2 = intval(((ord($hash2[0]) / 256) + (ord($hash2[1]) / (256 * 256)) + (ord($hash2[2]) / (256 * 256 * 256)) + (ord($hash2[3]) / (256 * 256 * 256 * 256))) * 6) + 1;
		$number3 = intval(((ord($hash3[0]) / 256) + (ord($hash3[1]) / (256 * 256)) + (ord($hash3[2]) / (256 * 256 * 256)) + (ord($hash3[3]) / (256 * 256 * 256 * 256))) * 6) + 1;
		return array($number1, $number2, $number3);	
	}
	
	public function get_last_game_data_by_uid($game_title, $user_id) {
		$sql = "SELECT * FROM `ez_games` WHERE `game_title` LIKE ? AND `user_id` = ? AND `completion_time` = '' ORDER BY `game_id` DESC LIMIT 1";
		$query = $this->db->query($sql, array($game_title, $user_id));
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_bet_info_by_server_seed($server_seed) {
		$sql = "SELECT * FROM `ez_games` WHERE `server_seed` = ? AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT 1";
		$query = $this->db->query($sql, array($server_seed));
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_bet_info_by_id($game_id) {
		$sql = "SELECT * FROM `ez_games` WHERE `game_id` = ? AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT 1";
		$query = $this->db->query($sql, array($game_id));
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function update_game_data($game_data, $updated_data) {
		$update_status = false;
		$data_changed = 0;
		foreach($game_data as $key => $value){
			if($updated_data->$key != $value) {
				$this->db->set($key, $updated_data->$key);
				$data_changed++;
			}
		}
		if($data_changed > 0) {
			$this->db->where('game_id', $game_data->game_id);
			$this->db->limit(1);
			$update_status = $this->db->update('ez_games');
			if(!$update_status) {
				throw new Exception("Server is busy");
			}
		}
		return $update_status;
	}

	public function get_last_claim_faucet_time($user_id, $ip) {
		$sql = "SELECT * FROM `ez_games` WHERE `game_title` = 'Faucet' AND (`user_id` = ? OR `ip` = ?) ORDER BY `created_time` DESC LIMIT 1";
		$query = $this->db->query($sql, array($user_id, $ip));
		if($query->num_rows() > 0) {
			return $query->result()[0]->created_time;
		}
		return 0;
	}
	
	public function load_user_bet_history($user_id, $page_id = 1, $game_title  = 'All') {
		if($user_id != 0) {
			if($game_title == 'All') {
				$sql = "SELECT `game_id`, `completion_time` AS `date`, `user_id`, `user_name`, `game_title` AS `game`, `nonce`, `client_seed`, `server_seed`, `bet_amount`, `payout`, `win_amount` AS `profit` FROM `ez_games` WHERE `user_id` = ? AND `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery' AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				$query = $this->db->query($sql, array($user_id));
			} else {
				$sql = "SELECT `game_id`, `completion_time` AS `date`, `user_id`, `user_name`, `game_title` AS `game`, `nonce`, `client_seed`, `server_seed`, `bet_amount`, `payout`, `win_amount` AS `profit`  FROM `ez_games` WHERE `user_id` = ? AND `game_title` = ? AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				$query = $this->db->query($sql, array($user_id, $game_title));
			}
			
		} else {
			if($game_title == 'All') {
				$sql = "SELECT `game_id`, `completion_time` AS `date`, `user_id`, `user_name`, `game_title` AS `game`, `nonce`, `client_seed`, `server_seed`, `bet_amount`, `payout`, `win_amount` AS `profit`  FROM `ez_games` WHERE `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery'  AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				$query = $this->db->query($sql);
			} else if($game_title == 'high_wins') {
				$sql = "SELECT `game_id`, `completion_time` AS `date`, `user_id`, `user_name`, `game_title` AS `game`, `bet_amount`, `payout`, `server_seed`, `win_amount` AS `profit`  FROM `ez_games` WHERE `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery'  AND `completion_time` != '' AND (`bet_amount` > 100 OR `win_amount` > 50) ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				$query = $this->db->query($sql);
			} else {
				$sql = "SELECT `game_id`, `completion_time` AS `date`, `user_id`, `user_name`, `game_title` AS `game`, `bet_amount`, `payout`, `server_seed`, `win_amount` AS `profit`  FROM `ez_games` WHERE `game_title` = ? AND `completion_time` != '' ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
				$query = $this->db->query($sql, array($game_title));
			}
		}
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}
	
	public function get_number_games($user_id = 0,  $game_title  = 'All') {
		if($user_id != 0) {
			if($game_title == 'All') {
				$sql = "SELECT COUNT(`game_id`) AS `total_games` FROM `ez_games` WHERE `user_id` = ? AND `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery' AND `completion_time` != ''";
				$query = $this->db->query($sql, array($user_id));
			} else {
				$sql = "SELECT COUNT(`game_id`) AS `total_games` FROM `ez_games` WHERE `user_id` = ? AND `game_title` = ? AND `completion_time` != ''";
				$query = $this->db->query($sql, array($user_id, $game_title));
			}
			
		} else {
			if($game_title == 'All') {
				$sql = "SELECT COUNT(`game_id`) AS `total_games` FROM `ez_games` WHERE `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery'  AND `completion_time` != ''";
				$query = $this->db->query($sql);
			} else {
				$sql = "SELECT COUNT(`game_id`) AS `total_games` FROM `ez_games` WHERE `game_title` = ? AND `completion_time` != ''";
				$query = $this->db->query($sql, array($game_title));
			}
		}
		if($query->num_rows() > 0) {
			return $query->result()[0]->total_games;
		}
		return 0;
	}
	
	public function get_statistics($user_id = 0) {
		if($user_id == 0) {
			$sql = "SELECT SUM(`bets`) AS `total_games`, SUM(`wagered`) AS `total_amount`, (SUM(`amount_losses`) - SUM(`amount_wins`)) AS `bet_balance` FROM `ez_users`";
			$query = $this->db->query($sql);
		} else {
			$sql = "SELECT SUM(`bets`) AS `total_games`, SUM(`wagered`) AS `total_amount`, (SUM(`amount_wins`) - SUM(`amount_losses`)) AS `bet_balance` FROM `ez_users` WHERE `user_id` = ?";
			$query = $this->db->query($sql, array($user_id));
		}
		
		
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function get_bet_statistics_today() {
		$start_of_today = strtotime('today');
		$sql = "SELECT COUNT(`user_id`) AS `total_users`, SUM(`bets_today`) AS `total_bets`, SUM(`bet_balance_today`) AS `bet_balance`, SUM(`wagered_today`) AS `total_wagered` FROM `ez_users` WHERE `ip_address` != '192.168.0.1' AND `last_bet_time` >= ?";
		$query = $this->db->query($sql, $start_of_today);
		if($query->num_rows() > 0) {
			return $query->result()[0];
		}
		return null;
	}
	
	public function load_lottery_tickets($user_id, $page_id = 1) {
		$this->limit_entries = 20;
		if($user_id != 0) {
			$sql = "SELECT `created_time`, `user_id`, `user_name`, `prediction` FROM `ez_games` WHERE `user_id` = ? AND `game_title` = 'Lottery' AND FROM_UNIXTIME(`created_time`, 'Y-M-D') = '".date('Y-m-d')."'  ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
			$query = $this->db->query($sql, array($user_id));
		} else {
			$sql = "SELECT `created_time`, `user_id`, `user_name`, `prediction` FROM `ez_games` WHERE `game_title` = 'Lottery' AND FROM_UNIXTIME(`created_time`, 'Y-M-D') = '".date('Y-m-d')."' ORDER BY `game_id` DESC LIMIT ".$this->limit_entries." OFFSET ".($this->limit_entries * ($page_id - 1));
			$query = $this->db->query($sql);
		}
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}
	
	public function load_faucet_claim_result() {
		$sql = "SELECT * FROM (SELECT `game_id`, `user_name`, `win_amount`, `completion_time`, `data` FROM `ez_games` WHERE `game_title` LIKE '%Faucet2%' AND completion_time > UNIX_TIMESTAMP() - 600 ORDER BY `win_amount` DESC LIMIT 20) AS T ORDER BY game_id DESC";
		$query = $this->db->query($sql);
		if($query->num_rows() > 0) {
			return $query->result();
		}
		return null;
	}
	
	public function delete_bet_rows_by_user($user_id, $deleted_rows) {
		$query = $this->db->query("DELETE FROM `ez_games` WHERE `user_id` = '".$user_id."' AND `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `completion_time` != '' AND `nonce` <= '".$deleted_rows."'");
	}
	
	public function clean_bet_data() {
		//$query = $this->db->query("DELETE FROM `ez_games` WHERE `game_title` != 'Faucet' AND `game_title` != 'Faucet2' AND `game_title` != 'Lottery' AND `completion_time` != '' AND `completion_time` < (UNIX_TIMESTAMP() - 86400)");
		$query = $this->db->query("DELETE FROM `ez_games` WHERE `win_amount` < 100000000 AND `completion_time` != '' AND `completion_time` < (UNIX_TIMESTAMP() - 86400) ORDER BY `completion_time` ASC LIMIT 1000");
	}
	
	//28/09/2023 cheat game gems reveal server_seed//
	public function allow_reveal_server_seed($server_seed) {
		$sql = "SELECT `game_id` FROM `ez_games` WHERE `server_seed` = ? AND `completion_time` = 0";
		$query = $this->db->query($sql, $server_seed);
		if($query->num_rows() > 0) {
			return 0;
		}
		return 1;
	}
	//28/09/2023 cheat game gems reveal server_seed//
	
	public function daily_game_market() {
		$sql = "SELECT `game_title` AS `game`, COUNT(`game_id`) AS `cnt` FROM `ez_games` GROUP BY `game` ORDER BY `cnt` ASC";
		$query = $this->db->query($sql);
		$labels = array();
		$data = array();
		if($query->num_rows() > 0) {
			$result = array_reverse($query->result());
			foreach($result as $row) {
				array_push($labels, $row->game);
				array_push($data, $row->cnt);
			}
		} 		
		return array(json_encode($labels), json_encode($data));
	}
	
	 public function get_daily_stats($start, $end) {
        $this->db->select("
            date,
            MAX(CASE WHEN statistic_key = 'users_bet' THEN statistic_value END) AS users_bet,
            MAX(CASE WHEN statistic_key = 'profit' THEN statistic_value END) AS profit,
            MAX(CASE WHEN statistic_key = 'games' THEN statistic_value END) AS games,
            MAX(CASE WHEN statistic_key = 'wagered' THEN statistic_value END) AS wagered
        ");
        $this->db->from('ez_statistics');
        $this->db->where_in('statistic_key', ['users_bet', 'profit', 'games', 'wagered']);
        $this->db->where("date >=", $start);
        $this->db->where("date <=", $end);
        $this->db->group_by('date');
        $this->db->order_by('date', 'ASC');
        return $this->db->get()->result_array();
    }

}
?>