<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

function generate_pcaptcha_image() {
    $CI =& get_instance();
    $CI->load->helper(['file']);
    $CI->load->library('session');

    // Xoá ảnh cũ hơn 5 phút
    $captchaDir = FCPATH . 'assets/captcha/';
    $files = glob($captchaDir . 'captcha_*.jpg');
    $now = time();
    foreach ($files as $file) {
        if (is_file($file) && ($now - filemtime($file)) > 300) {
            @unlink($file);
        }
    }

    // Load câu hỏi & nhãn
    $captchaData = json_decode(read_file(FCPATH . 'assets/captcha/captcha_questions.json'), true);
    $labels = json_decode(read_file(FCPATH . 'assets/captcha/labels.json'), true);
    $questions = $captchaData['questions'];

    // Chọn câu hỏi ngẫu nhiên
    $question = $questions[array_rand($questions)];
    $correct_answers = $question['correct_answer'];

    $correctImages = [];
    $otherImages = [];

    foreach ($labels as $imagePath => $label) {
        $imagePath = str_replace('\\', '/', $imagePath);
        $fullPath = FCPATH . 'assets/captcha/dataset/' . $imagePath;

        $label_clean = strtolower(trim($label));
        $correct_clean = array_map('strtolower', array_map('trim', $correct_answers));

        if (file_exists($fullPath)) {
            if (in_array($label_clean, $correct_clean)) {
                $correctImages[] = $imagePath;
            } else {
                $otherImages[] = $imagePath;
            }
        }
    }

    if (count($correctImages) < 2) {
        return ['error' => 'Not enough correct images'];
    }

    $numCorrect = (count($correctImages) >= 3) ? rand(2, 3) : 2;
    shuffle($correctImages);
    shuffle($otherImages);

    $selectedCorrect = array_slice($correctImages, 0, $numCorrect);
    $selectedWrong = array_slice($otherImages, 0, 9 - $numCorrect);
    $allImages = array_merge($selectedCorrect, $selectedWrong);
    shuffle($allImages);

    $cellSize = 100;
    $gridSize = 3;
    $finalImage = imagecreatetruecolor($cellSize * $gridSize, $cellSize * $gridSize);

    $correctIndices = [];

    for ($i = 0; $i < count($allImages); $i++) {
        if (in_array($allImages[$i], $selectedCorrect)) {
            $correctIndices[] = $i;
        }

        $fullPath = FCPATH . 'assets/captcha/dataset/' . $allImages[$i];
        $ext = strtolower(pathinfo($fullPath, PATHINFO_EXTENSION));
        if ($ext === 'jpg' || $ext === 'jpeg') {
            $src = @imagecreatefromjpeg($fullPath);
        } elseif ($ext === 'png') {
            $src = @imagecreatefrompng($fullPath);
        } else {
            continue;
        }

        if (!$src) continue;

        $resized = imagecreatetruecolor($cellSize, $cellSize);
        imagecopyresampled($resized, $src, 0, 0, 0, 0, $cellSize, $cellSize, imagesx($src), imagesy($src));

        // Các hiệu ứng tùy chọn
        $effects = [
            'none' => fn($img) => $img,
            'circle' => 'applyCirclePattern',
            'square' => 'applySquarePattern',
            'blur' => 'applyBlur',
            'grayscale' => 'applyGrayscale',
            'circle_blur' => fn($img) => applyBlur(applyCirclePattern($img)),
            'square_grayscale' => fn($img) => applyGrayscale(applySquarePattern($img))
        ];
        $effect = array_rand($effects);
        $processed = $effects[$effect]($resized);

        $row = floor($i / $gridSize);
        $col = $i % $gridSize;
        imagecopy($finalImage, $processed, $col * $cellSize, $row * $cellSize, 0, 0, $cellSize, $cellSize);

        imagedestroy($src);
        if ($processed !== $resized) imagedestroy($processed);
        imagedestroy($resized);
    }

    // Tạo tên ảnh theo session
    $filename = 'captcha_' . session_id() . '.jpg';
    $outputPath = $captchaDir . $filename;
    imagejpeg($finalImage, $outputPath, 85);
    imagedestroy($finalImage);

    // Ghi session
    $CI->session->set_userdata('pcaptcha_answer', $correctIndices);
    $CI->session->set_userdata('pcaptcha_question', $question['question']);
    $CI->session->set_userdata('pcaptcha_target', $correct_answers);
    $CI->session->set_userdata('pcaptcha_image', $filename);

    // Đảm bảo ghi session nếu dùng memcached
    if (method_exists($CI->session, 'sess_write')) {
        $CI->session->sess_write();
    }

    return [
        'question' => $question['question'],
        'image' => base_url() . 'assets/captcha/' . $filename . '?t=' . time()
    ];
}

function verify_pcaptcha($selected_boxes = '[]') {
    $CI =& get_instance();
    $CI->load->library('session');

    $max_attempts = 10;
    $wait_time_seconds = 300; // 5 phút
    $current_time = time();

    $fail_count = (int) ($CI->session->userdata('pcaptcha_fail_count') ?? 0);
    $last_fail_time = (int) ($CI->session->userdata('pcaptcha_last_fail_time') ?? 0);

    if ($fail_count >= $max_attempts && ($current_time - $last_fail_time) < $wait_time_seconds) {
        $remaining = $wait_time_seconds - ($current_time - $last_fail_time);
        return [
            'success' => false,
            'message' => 'Too many failed attempts. Please wait ' . ceil($remaining / 60) . ' minutes before trying again.',
            'token' => null
        ];
    }

    $correct = $CI->session->userdata('pcaptcha_answer') ?? [];
    $user = json_decode($selected_boxes, true);

    sort($correct);
    sort($user);

    $isCorrect = ($correct === $user);
    $message = '';
    $token = null;

    if (!$isCorrect) {
        $fail_count++;
        $CI->session->set_userdata('pcaptcha_fail_count', $fail_count);
        $CI->session->set_userdata('pcaptcha_last_fail_time', $current_time);

        $correctCount = count($correct);
        $userCount = count($user);

        if ($userCount === 0) {
            $message = 'Please select all images that match the description.';
        } elseif ($userCount < $correctCount) {
            $missing = $correctCount - $userCount;
            $message = $missing === 1
                ? 'You missed one image. Please select all matching images.'
                : "You missed $missing images. Please select all matching images.";
        } elseif ($userCount > $correctCount) {
            $extra = $userCount - $correctCount;
            $message = $extra === 1
                ? 'You selected one extra image. Please try again.'
                : "You selected $extra extra images. Please try again.";
        } else {
            $message = 'Incorrect selection. Please try again.';
        }
    } else {
        $message = 'Success!';
        $token = 'captcha_' . session_id() . '_' . time() . '_' . bin2hex(random_bytes(8));

        $CI->session->set_userdata('pcaptcha_token', $token);
        $CI->session->set_userdata('pcaptcha_verified', true);
        $CI->session->set_userdata('pcaptcha_verified_time', $current_time);

        // Nếu bạn dùng memcached hoặc driver khác
        if (method_exists($CI->session, 'sess_write')) {
            $CI->session->sess_write();
        }

        // Xoá ảnh cũ
        $captcha_image = $CI->session->userdata('pcaptcha_image');
        if (!empty($captcha_image) && file_exists(FCPATH . 'assets/captcha/' . $captcha_image)) {
            @unlink(FCPATH . 'assets/captcha/' . $captcha_image);
        }

        // Dọn phần không cần
        $CI->session->unset_userdata('pcaptcha_fail_count');
        $CI->session->unset_userdata('pcaptcha_last_fail_time');
        $CI->session->unset_userdata('pcaptcha_answer');
        $CI->session->unset_userdata('pcaptcha_question');
        $CI->session->unset_userdata('pcaptcha_target');
        $CI->session->unset_userdata('pcaptcha_image');
        // ❌ KHÔNG XÓA: token, verified, verified_time
    }

    return [
        'success' => $isCorrect,
        'message' => $message,
        'token'   => $token
    ];
}

function render_pcaptcha_widget($id = 'pcaptcha1', $form_id = 'myform') {
    $base_url = base_url();
    $safe_id = htmlspecialchars($id, ENT_QUOTES);
    $safe_form_id = htmlspecialchars($form_id, ENT_QUOTES);

    echo <<<HTML
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<link rel="stylesheet" href="{$base_url}assets/css/pcaptcha.css">

<!-- CAPTCHA Widget -->
<input type="hidden" name="pcaptcha_token" id="pcaptcha-token-hidden">
<div class="captcha-container" id="captcha-container-{$safe_id}">
  <div class="captcha-checkbox-container" id="checkbox-container-{$safe_id}">
    <div class="captcha-checkbox-left">
      <div class="captcha-checkbox" id="checkbox-{$safe_id}"><span id="checkbox-icon-{$safe_id}"></span></div>
      <span class="captcha-checkbox-text" id="checkbox-text-{$safe_id}">I'm not a robot</span>
    </div>
    <div class="captcha-recaptcha-logo">
      <div class="captcha-logo-icon">P</div>
      <div class="captcha-logo-text">CAPTCHA</div>
    </div>
  </div>

  <div class="captcha-challenge" id="captcha-challenge-{$safe_id}" style="display:none;">
    <div class="captcha-challenge-header" id="challenge-question-{$safe_id}">Select all images containing...</div>
    <div class="captcha-challenge-content">
      <div class="captcha-image-grid" id="image-grid-{$safe_id}"></div>
      <div class="captcha-challenge-controls">
        <button class="captcha-control-btn" id="refresh-btn-{$safe_id}">↻</button>
        <button class="captcha-verify-btn" id="verify-btn-{$safe_id}" disabled>Verify</button>
      </div>
    </div>
  </div>

  <div class="captcha-error-message" id="error-message-{$safe_id}"></div>
</div>

<script>
(function(){
    const BASE_URL = '{$base_url}';
    const id = '{$safe_id}';
    const selected = new Set();
    let isVerified = false;
    let currentImagePath = null;

    const \$container = $('#captcha-container-' + id);
    const \$checkboxContainer = $('#checkbox-container-' + id);
    const \$checkbox = $('#checkbox-' + id);
    const \$checkboxIcon = $('#checkbox-icon-' + id);
    const \$checkboxText = $('#checkbox-text-' + id);
    const \$challenge = $('#captcha-challenge-' + id);
    const \$question = $('#challenge-question-' + id);
    const \$imageGrid = $('#image-grid-' + id);
    const \$refreshBtn = $('#refresh-btn-' + id);
    const \$verifyBtn = $('#verify-btn-' + id);
    const \$errorMsg = $('#error-message-' + id);
    const \$tokenInput = $('#pcaptcha-token-hidden');

    function loadCaptcha() {
        hideError();
        $.ajax({
            url: BASE_URL + "generate_pcaptcha.php",
            method: "GET",
            dataType: "json",
            success: function (data) {
                if (data.error) {
                    showError('Error: ' + data.error);
                    return;
                }

                \$question.html(data.question);
                currentImagePath = data.image;
                loadImages(currentImagePath);
                selected.clear();
                \$verifyBtn.prop('disabled', true);
            },
            error: function () {
                showError('Error loading challenge. Please try again.');
            }
        });
    }

    function bindImageClicks() {
        \$imageGrid.off('click').on('click', '.captcha-image-cell', function () {
            const index = parseInt($(this).data('index'));
            if ($(this).hasClass('selected')) {
                selected.delete(index);
                $(this).removeClass('selected');
            } else {
                selected.add(index);
                $(this).addClass('selected');
            }
            \$verifyBtn.prop('disabled', selected.size === 0);
        });
    }

    function loadImages(imageSrc) {
        \$imageGrid.empty();
        for (let i = 0; i < 9; i++) {
            const row = Math.floor(i / 3);
            const col = i % 3;
            const cell = $('<div>')
                .addClass('captcha-image-cell')
                .attr('data-index', i)
                .html('<div class="captcha-loading-text">Loading...</div>')
                .css({
                    'background-image': `url(\${imageSrc}&part=\${i})`,
                    'background-position': `\${-col * 100}px \${-row * 100}px`,
                    'background-size': '300px 300px',
                    'background-repeat': 'no-repeat'
                });

            \$imageGrid.append(cell);
        }

        bindImageClicks();

        const testImg = new Image();
        testImg.onload = () => $('.captcha-loading-text').fadeOut();
        testImg.onerror = () => $('.captcha-loading-text').text('Error').css('color', 'red');
        testImg.src = imageSrc + '&check=' + Date.now();
    }

    \$checkboxContainer.on('click', function () {
        if (!isVerified) {
            \$checkbox.addClass('animating');
            for (let i = 1; i <= 3; i++) {
                setTimeout(() => {
                    \$checkbox.append(`<div class="captcha-ripple-circle captcha-ripple-\${i}"></div>`);
                }, i * 100);
            }

            setTimeout(() => {
                loadCaptcha();
                \$checkboxContainer.addClass('verified');
                \$checkbox.addClass('checked');
                \$checkboxIcon.text('✓');
                \$challenge.show();
                \$checkboxContainer.hide();
                $('.captcha-ripple-circle').remove();
                \$checkbox.removeClass('animating');
            }, 800);
        }
    });

    \$refreshBtn.on('click', function () {
        loadCaptcha();
    });

    \$verifyBtn.on('click', function () {
        if (selected.size === 0) {
            showError('Please select at least one image before verifying.');
            return;
        }

        \$verifyBtn.prop('disabled', true).text('Verifying...');

        $.ajax({
            url: BASE_URL + "process.php",
            method: "POST",
            data: {
                action: 'verify_pcaptcha',
                selected_boxes: JSON.stringify([...selected])
            },
            dataType: "json",
            success: function (response) {
                if (response.success) {
                    isVerified = true;
                    \$challenge.hide();
                    \$checkboxContainer.show().addClass('verified');
                    \$checkbox.addClass('verified');
                    \$checkboxIcon.text('✓');
                    \$checkboxText.text('Verified successfully!');
                    \$container.attr('data-token', response.token);
                    \$tokenInput.val(response.token);
                } else {
                    showError(response.message || 'Incorrect selection. Please try again.');
                    loadCaptcha();
                }
            },
            error: function () {
                showError('Error during verification. Please try again.');
            },
            complete: function () {
                \$verifyBtn.prop('disabled', false).text('Verify');
            }
        });
    });

    function showError(message) {
        \$errorMsg.text(message).css('color', 'red').show();
    }

    function hideError() {
        \$errorMsg.hide().text('');
    }

    // ✅ Reset CAPTCHA
    window.resetPCaptcha = window.resetPCaptcha || function (captchaId) {
        const \$box = $('#captcha-container-' + captchaId);
        const \$checkboxContainer = \$box.find('.captcha-checkbox-container');
        const \$checkbox = \$box.find('.captcha-checkbox');
        const \$checkboxIcon = \$box.find('.captcha-checkbox span');
        const \$checkboxText = \$box.find('.captcha-checkbox-text');
        const \$challenge = \$box.find('.captcha-challenge');

        \$checkboxContainer.show().removeClass('verified');
        \$checkbox.removeClass('verified checked');
        \$checkboxIcon.text('');
        \$checkboxText.text("I'm not a robot");
        \$challenge.hide();
        \$box.removeAttr('data-token');
        \$tokenInput.val('');
        isVerified = false;
    };

})();
</script>
HTML;
}

// Hàm tạo hiệu ứng vân mờ (concentric circles)
function applyCirclePattern($image) {
    $width = imagesx($image);
    $height = imagesy($image);
    $pattern = imagecreatetruecolor($width, $height);
    
    // Tạo pattern vòng tròn đồng tâm
    for ($x = 0; $x < $width; $x++) {
        for ($y = 0; $y < $height; $y++) {
            $centerX = $width / 2;
            $centerY = $height / 2;
            $distance = sqrt(pow($x - $centerX, 2) + pow($y - $centerY, 2));
            $frequency = 0.3; // Giảm tần số vòng tròn
            
            $intensity = sin($distance * $frequency) * 0.15 + 0.85; // Giảm độ mờ
            $intensity = max(0, min(1, $intensity));
            
            $rgb = imagecolorat($image, $x, $y);
            $r = ($rgb >> 16) & 0xFF;
            $g = ($rgb >> 8) & 0xFF;
            $b = $rgb & 0xFF;
            
            $r = (int)($r * $intensity);
            $g = (int)($g * $intensity);
            $b = (int)($b * $intensity);
            
            $color = imagecolorallocate($pattern, $r, $g, $b);
            imagesetpixel($pattern, $x, $y, $color);
        }
    }
    
    return $pattern;
}

// Hàm tạo hiệu ứng vân ô vuông
function applySquarePattern($image) {
    $width = imagesx($image);
    $height = imagesy($image);
    $pattern = imagecreatetruecolor($width, $height);
    
    // Tạo pattern ô vuông đồng tâm
    for ($x = 0; $x < $width; $x++) {
        for ($y = 0; $y < $height; $y++) {
            $centerX = $width / 2;
            $centerY = $height / 2;
            $distance = max(abs($x - $centerX), abs($y - $centerY));
            $frequency = 0.3;
            
            $intensity = sin($distance * $frequency) * 0.2 + 0.8;
            $intensity = max(0, min(1, $intensity));
            
            $rgb = imagecolorat($image, $x, $y);
            $r = ($rgb >> 16) & 0xFF;
            $g = ($rgb >> 8) & 0xFF;
            $b = $rgb & 0xFF;
            
            $r = (int)($r * $intensity);
            $g = (int)($g * $intensity);
            $b = (int)($b * $intensity);
            
            $color = imagecolorallocate($pattern, $r, $g, $b);
            imagesetpixel($pattern, $x, $y, $color);
        }
    }
    
    return $pattern;
}

// Hàm làm mờ ảnh
function applyBlur($image) {
    $width = imagesx($image);
    $height = imagesy($image);
    $blurred = imagecreatetruecolor($width, $height);
    
    // Kernel blur 3x3
    $kernel = [
        [1, 1, 1],
        [1, 1, 1],
        [1, 1, 1]
    ];
    $divisor = 9;
    
    for ($x = 1; $x < $width - 1; $x++) {
        for ($y = 1; $y < $height - 1; $y++) {
            $r = $g = $b = 0;
            
            for ($i = -1; $i <= 1; $i++) {
                for ($j = -1; $j <= 1; $j++) {
                    $rgb = imagecolorat($image, $x + $i, $y + $j);
                    $r += (($rgb >> 16) & 0xFF) * $kernel[$i + 1][$j + 1];
                    $g += (($rgb >> 8) & 0xFF) * $kernel[$i + 1][$j + 1];
                    $b += ($rgb & 0xFF) * $kernel[$i + 1][$j + 1];
                }
            }
            
            $r = (int)($r / $divisor);
            $g = (int)($g / $divisor);
            $b = (int)($b / $divisor);
            
            $color = imagecolorallocate($blurred, $r, $g, $b);
            imagesetpixel($blurred, $x, $y, $color);
        }
    }
    
    return $blurred;
}

// Hàm chuyển ảnh sang đen trắng
function applyGrayscale($image) {
    $width = imagesx($image);
    $height = imagesy($image);
    $grayscale = imagecreatetruecolor($width, $height);
    
    for ($x = 0; $x < $width; $x++) {
        for ($y = 0; $y < $height; $y++) {
            $rgb = imagecolorat($image, $x, $y);
            $r = ($rgb >> 16) & 0xFF;
            $g = ($rgb >> 8) & 0xFF;
            $b = $rgb & 0xFF;
            
            $gray = (int)(0.299 * $r + 0.587 * $g + 0.114 * $b);
            
            $color = imagecolorallocate($grayscale, $gray, $gray, $gray);
            imagesetpixel($grayscale, $x, $y, $color);
        }
    }
    
    return $grayscale;
}